'------------------------------------------------------------------------------
' <copyright from='1997' to='2002' company='Microsoft Corporation'>
'    Copyright (c) Microsoft Corporation. All Rights Reserved.
'
'    This source code is intended only as a supplement to Microsoft
'    Development Tools and/or on-line documentation.  See these other
'    materials for detailed information regarding Microsoft code samples.
'
' </copyright>
'------------------------------------------------------------------------------
Imports System
Imports System.ComponentModel.Design
Imports System.Collections
Imports System.Reflection


''' The ITypeResolutionService is used to load types at design time. It keeps an internal
''' set of assemblies that have been referenced thus far and can search them for types given
''' unqualified names.
Friend Class TypeResolutionService
    Implements System.ComponentModel.Design.ITypeResolutionService
    Private assemblies As Hashtable
    Public Sub New()
    End Sub

    ''' We use this property to help us generate code and compile.
    Public ReadOnly Property RefencedAssemblies() As Assembly()
        Get
            If assemblies Is Nothing Then
                assemblies = New Hashtable()
            End If
            Dim ret As Assembly() = New Assembly(assemblies.Values.Count - 1) {}
            assemblies.Values.CopyTo(ret, 0)
            Return ret
        End Get
    End Property

#Region "Implementation of ITypeResolutionService"

    ''' Add an assembly to our internal set.
    Public Sub ReferenceAssembly(ByVal name As System.Reflection.AssemblyName) Implements ITypeResolutionService.ReferenceAssembly
        If assemblies Is Nothing Then
            assemblies = New Hashtable()
        End If

        If Not assemblies.Contains(name) Then
            assemblies.Add(name, Assembly.Load(name))
        End If
    End Sub

    ''' Search our internal set of assemblies for one with this AssemblyName.
    ''' If it cannot be located and throwOnError is true, throw an exception.
    Public Function GetAssembly(ByVal name As System.Reflection.AssemblyName, ByVal throwOnError As Boolean) As System.Reflection.Assembly Implements ITypeResolutionService.GetAssembly
        Dim a As Assembly = Nothing

        If assemblies IsNot Nothing Then
            a = TryCast(assemblies(name), Assembly)
        End If

        If (a Is Nothing) AndAlso throwOnError Then
            Throw New Exception("Assembly " & name.Name & " not found in referenced assemblies.")
        End If
        Return a
    End Function

    ''' Search our internal set of assemblies for one with this AssemblyName.
    Private Function GetAssembly(ByVal name As System.Reflection.AssemblyName) As System.Reflection.Assembly Implements System.ComponentModel.Design.ITypeResolutionService.GetAssembly
        Return GetAssembly(name, False)
    End Function

    ''' Find a type based on a name that may or may not be fully qualified.
    ''' If the type cannot be found and throwOnError is true, throw an Exception.
    ''' Searching should ignore case based on ignoreCase's value.
    Public Overloads Function [GetType](ByVal name As String, ByVal throwOnError As Boolean, ByVal ignoreCase As Boolean) As Type Implements ITypeResolutionService.GetType
        ' First we assume it is a fully qualified name, and that Type.GetType()
        ' can find it.
        '
        Dim t As Type = Type.[GetType](name, throwOnError, ignoreCase)

        ' If that didn't work, then we check our referenced assemblies' types.
        If (t Is Nothing) AndAlso (assemblies IsNot Nothing) Then
            For Each a As Assembly In assemblies.Values
                t = a.[GetType](name, throwOnError, ignoreCase)
                If t IsNot Nothing Then
                    Exit For
                End If
            Next
        End If

        ' No luck.
        If (t Is Nothing) AndAlso throwOnError Then
            Throw New Exception("The type " & name & " could not be found. " & "If it is an unqualified name, then its assembly has not been referenced.")
        End If
        Return t
    End Function

    ''' Find a type based on a name that may or may not be fully qualified.
    ''' If the type cannot be found and throwOnError is true, throw an Exception.
    ''' Do not ignore case while searching.
    Private Overloads Function [GetType](ByVal name As String, ByVal throwOnError As Boolean) As Type Implements System.ComponentModel.Design.ITypeResolutionService.[GetType]
        Return [GetType](name, throwOnError, False)
    End Function

    ''' Find a type based on a name that may or may not be fully qualified.
    ''' Do not ignore case while searching.
    Private Overloads Function [GetType](ByVal name As String) As Type Implements System.ComponentModel.Design.ITypeResolutionService.[GetType]
        Return [GetType](name, False, False)
    End Function

    ''' Return the path to the file which the given assembly was loaded.
    ''' If that assembly has not been loaded, this returns null.
    Public Function GetPathOfAssembly(ByVal name As System.Reflection.AssemblyName) As String Implements ITypeResolutionService.GetPathOfAssembly
        Dim a As Assembly = GetAssembly(name, False)
        If a IsNot Nothing Then
            Return a.Location
        End If
        Return Nothing
    End Function

#End Region
End Class


