'==============================================================================
' Basiert auf der Arbeit von Tim Dawson, www.divil.co.uk
'==============================================================================
Imports System
Imports System.Collections
Imports System.ComponentModel
Imports System.ComponentModel.Design
Imports System.Windows.Forms


Friend Class SelectionService
    Implements ISelectionService

    Public Event SelectionChanging As EventHandler Implements ISelectionService.SelectionChanging
    Public Event SelectionChanged As EventHandler Implements ISelectionService.SelectionChanged

    Private _host As IDesignerHost = Nothing
    Private selectedComponents As ArrayList = New ArrayList()

    Public Sub New(ByVal host As IDesignerHost)
        _host = host

        selectedComponents = New ArrayList()

        ' Subscribe to the componentremoved event
        Dim c As IComponentChangeService = CType(host.GetService(GetType(IComponentChangeService)), IComponentChangeService)
        AddHandler c.ComponentRemoved, AddressOf OnComponentRemoved
    End Sub

    Public Function GetSelectedComponents() As ICollection Implements ISelectionService.GetSelectedComponents
        If Not selectedComponents Is Nothing Then
            Return selectedComponents.ToArray()
        Else
            Return Nothing
        End If
    End Function

    Friend Sub OnSelectionChanging(ByVal e As EventArgs)
        ' Fire the SelectionChanging event if anything is bound to it
        RaiseEvent SelectionChanging(Me, e)
    End Sub

    Friend Sub OnSelectionChanged(ByVal e As EventArgs)
        If EventArgs.Empty.ToString = "System.EventArgs" Then Exit Sub
        ' Fire the SelectionChanged event if anything is bound to it
        RaiseEvent SelectionChanged(Me, e)
    End Sub

    Public ReadOnly Property PrimarySelection() As Object Implements ISelectionService.PrimarySelection
        Get
            ' If we have any selected components, return the first
            If Not selectedComponents Is Nothing Then
                If (selectedComponents.Count > 0) Then
                    Return selectedComponents(0)
                Else
                    Return Nothing
                End If
            Else
                Return Nothing
            End If
        End Get
    End Property

    Public ReadOnly Property SelectionCount() As Integer Implements ISelectionService.SelectionCount
        Get
            If Not selectedComponents Is Nothing Then
                Return selectedComponents.Count
            Else
                Return 0
            End If
        End Get
    End Property

    Public Function GetComponentSelected(ByVal component As Object) As Boolean Implements ISelectionService.GetComponentSelected
        Return selectedComponents.Contains(component)
    End Function

    Public Sub SetSelectedComponents(ByVal components As ICollection, ByVal selectionType As SelectionTypes) Implements ISelectionService.SetSelectedComponents
        Dim control As Boolean = False
        Dim shift As Boolean = False

        ' Raise selectionchanging event
        RaiseEvent SelectionChanging(Me, EventArgs.Empty)

        ' If we're being passed an empty collection
        If (components Is Nothing Or components.Count = 0) Then
            Dim c() As IComponent = {_host.RootComponent}
            components = New ComponentCollection(c)
        End If

        ' If the selection type is Click, we want to know if shift or control is being held
        If ((selectionType And SelectionTypes.Primary) = SelectionTypes.Primary) Then
            control = ((System.Windows.Forms.Control.ModifierKeys And Keys.Control) = Keys.Control)
            shift = ((System.Windows.Forms.Control.ModifierKeys And Keys.Shift) = Keys.Shift)
        End If

        If (selectionType = SelectionTypes.Replace) Then
            ' Simply replace our existing collection with the new one
            selectedComponents.Clear()
            Dim component As Object
            For Each component In components
                If ((Not component Is Nothing) And (Not selectedComponents.Contains(component))) Then
                    selectedComponents.Add(component)
                End If
            Next
        Else
            ' Clear selection if ctrl or shift isn't pressed
            Dim component As Object
            If (Not control And Not shift And components.Count = 1) Then
                For Each component In components
                    If (Not selectedComponents.Contains(component)) Then
                        selectedComponents.Clear()
                    End If
                Next
            End If

            ' Add or remove each component to or from the selection
            For Each component In components
                If (Not component Is Nothing) Then
                    If (control Or shift) Then
                        If (selectedComponents.Contains(component)) Then
                            selectedComponents.Remove(component)
                        Else
                            selectedComponents.Insert(0, component)
                        End If
                    Else
                        If (Not selectedComponents.Contains(component)) Then
                            selectedComponents.Add(component)
                        Else
                            selectedComponents.Remove(component)
                            selectedComponents.Insert(0, component)
                        End If
                    End If
                End If
            Next
        End If

        Try
            ' Raise the selectionchanged event
            RaiseEvent SelectionChanged(Me, EventArgs.Empty)

        Catch
            MsgBox("Why am I getting an error when raising the SelectionChanged event?", MsgBoxStyle.Exclamation)
        End Try
    End Sub

    Public Sub SetSelectedComponents(ByVal components As ICollection) Implements ISelectionService.SetSelectedComponents
        ' Use the Replace selection type because this needs to replace anything already selected
        SetSelectedComponents(components, SelectionTypes.Replace)
    End Sub

    Friend Sub OnComponentRemoved(ByVal sender As Object, ByVal e As ComponentEventArgs)
        If (selectedComponents.Contains(e.Component)) Then
            ' Raise the selectionchanging event
            OnSelectionChanging(EventArgs.Empty)

            ' Remove this component from the selected components
            selectedComponents.Remove(e.Component)

            ' Select root component if that leaves us with no selected components
            If (SelectionCount = 0) Then
                selectedComponents.Add(_host.RootComponent)
            End If

            ' Raise the selectionchanged event
            OnSelectionChanged(EventArgs.Empty)
        End If
    End Sub
End Class
