'------------------------------------------------------------------------------
' <copyright from='1997' to='2002' company='Microsoft Corporation'>
'    Copyright (c) Microsoft Corporation. All Rights Reserved.
'
'    This source code is intended only as a supplement to Microsoft
'    Development Tools and/or on-line documentation.  See these other
'    materials for detailed information regarding Microsoft code samples.
'
' </copyright>
'------------------------------------------------------------------------------
Imports System
Imports System.Collections
Imports System.ComponentModel
Imports System.ComponentModel.Design
'Imports System.Design
Imports System.Diagnostics
Imports System.Globalization
Imports System.Text



'This class provides a default implementation of the event
'binding service.
Friend Class EventBindingService
    Implements IEventBindingService
    Private _eventProperties As Hashtable
    Private _provider As IServiceProvider

    'You must provide a service provider to the binding
    'service. We give it our host.
    Public Sub New(ByVal provider As IServiceProvider)
        If provider Is Nothing Then
            Throw New ArgumentNullException("provider")
        End If
        _provider = provider
    End Sub

    'Creates a unique method name.  The name must be
    'compatible with the script language being used and
    'it must not conflict with any other name in the user's
    'code. Since we have no code editor, we can guarantee this
    'method name will be unique. However, if code were editable,
    'we would have to check our codeCompileUnit's methods.
    'Protected Function CreateUniqueMethodName(ByVal component As IComponent, ByVal e As EventDescriptor) As String
    '    Dim name As String = component.Site.Name.ToString().Split(New Char() {" "c})(0)
    '    Return ("handler_" & name & "_") + e.Name
    'End Function

    'This provides a notification that a particular method
    'is no longer being used by an event handler.  Some implementations
    'may want to remove the event hander when no events are using
    'it.  By overriding UseMethod and FreeMethod, an implementation
    'can know when a method is no longer needed.
    Protected Overridable Sub FreeMethod(ByVal component As Object, ByVal e As EventDescriptor, ByVal methodName As String)
        ' UNIMPLEMENTED - We don't add method signatures for our handlers, so
        ' don't need to worry about removing them.
    End Sub

    'Returns a collection of strings.  Each string is
    'the method name of a method whose signature is
    'compatible with the delegate contained in the
    'event descriptor.  This should return an empty
    'collection if no names are compatible.
    'Protected Function GetCompatibleMethods(ByVal e As EventDescriptor) As ICollection
    '    ' EMPTY IMPLEMENTATION
    '    Return New String() {}
    'End Function

    'Generates a key based on a method name and it's parameters by just concatenating the
    'parameters.
    Private Function GetEventDescriptorHashCode(ByVal eventDesc As EventDescriptor) As String
        Dim builder As New StringBuilder(eventDesc.Name)
        builder.Append(eventDesc.EventType.GetHashCode().ToString())

        For Each a As Attribute In eventDesc.Attributes
            builder.Append(a.GetHashCode().ToString())
        Next

        Return builder.ToString()
    End Function

    'Gets the requested service from our service provider (the host).	
    Protected Function GetService(ByVal serviceType As Type) As Object
        If _provider IsNot Nothing Then
            Return _provider.GetService(serviceType)
        End If
        Return Nothing
    End Function

    'Shows the user code.  This method does not show any
    'particular code; generally it shows the last code the
    'user typed.  This returns true if it was possible to 
    'show the code, or false if not. We are never showing code
    'since we do not generate handler methods.
    'Protected Function ShowCode() As Boolean
    '    Return False
    'End Function

    'Shows the user code at the given line number.  Line
    'numbers are one-based.  This returns true if it was
    'possible to show the code, or false if not. We are 
    'never showing code since we do not generate handler methods.
    'Protected Function ShowCode(ByVal lineNumber As Integer) As Boolean
    '    Return False
    'End Function

    'Shows the body of the user code with the given method
    'name. This returns true if it was possible to show
    'the code, or false if not. We are never showing code
    'since we do not generate handler methods.
    Protected Function ShowCode(ByVal component As Object, ByVal e As EventDescriptor, ByVal methodName As String) As Boolean
        Return False
    End Function

    'This provides a notification that a particular method
    'is being used by an event handler.  Some implementations
    'may want to remove the event hander when no events are using
    'it.  By overriding UseMethod and FreeMethod, an implementation
    'can know when a method is no longer needed.
    Protected Overridable Sub UseMethod(ByVal component As Object, ByVal e As EventDescriptor, ByVal methodName As String)
        ' UNIMPLEMENTED - We do not add method signatures to our code.
    End Sub

    'This validates that the provided method name is valid for
    'the language / script being used.  The default does nothing.
    'You may override this and throw an exception if the name
    'is invalid for your use.
    Protected Overridable Sub ValidateMethodName(ByVal methodName As String)
        ' UNIMPLEMENTED - We are guaranteed our method names are valid.
    End Sub

    'This creates a name for an event handling method for the given component
    'and event.  The name that is created is guaranteed to be unique in the user's source
    'code.
    Private Function CreateUniqueMethodName(ByVal component As IComponent, ByVal e As EventDescriptor) As String Implements IEventBindingService.CreateUniqueMethodName
        If component Is Nothing Then
            Throw New ArgumentNullException("component")
        End If

        If e Is Nothing Then
            Throw New ArgumentNullException("e")
        End If

        Return CreateUniqueMethodName(component, e)
    End Function

    'Retrieves a collection of strings.  Each string is the name of a method
    'in user code that has a signature that is compatible with the given event.
    Private Function GetCompatibleMethods(ByVal e As EventDescriptor) As ICollection Implements IEventBindingService.GetCompatibleMethods
        If e Is Nothing Then
            Throw New ArgumentNullException("e")
        End If

        Return GetCompatibleMethods(e)
    End Function

    'For properties that are representing events, this will return the event
    'that the property represents.
    Private Function GetEvent(ByVal [property] As PropertyDescriptor) As EventDescriptor Implements IEventBindingService.GetEvent
        If TypeOf [property] Is EventPropertyDescriptor Then
            Return DirectCast([property], EventPropertyDescriptor).[Event]
        Else
            Return Nothing
        End If
    End Function

    'Converts a set of events to a set of properties.
    Private Function GetEventProperties(ByVal events As EventDescriptorCollection) As PropertyDescriptorCollection Implements IEventBindingService.GetEventProperties

        If events Is Nothing Then
            Throw New ArgumentNullException("events")
        End If

        Dim props As PropertyDescriptor() = New PropertyDescriptor(events.Count - 1) {}

        ' We cache the property descriptors here for speed.  Create those for
        ' events that we don't have yet.
        '
        If _eventProperties Is Nothing Then
            _eventProperties = New Hashtable()
        End If

        For i As Integer = 0 To events.Count - 1

            Dim eventHashCode As Object = GetEventDescriptorHashCode(events(i))

            props(i) = DirectCast(_eventProperties(eventHashCode), PropertyDescriptor)

            If props(i) Is Nothing Then
                props(i) = New EventPropertyDescriptor(events(i), Me)
                _eventProperties(eventHashCode) = props(i)
            End If
        Next

        Return New PropertyDescriptorCollection(props)
    End Function

    'Converts a single event to a property.
    Private Function GetEventProperty(ByVal e As EventDescriptor) As PropertyDescriptor Implements IEventBindingService.GetEventProperty

        If e Is Nothing Then
            Throw New ArgumentNullException("e")
        End If

        If _eventProperties Is Nothing Then
            _eventProperties = New Hashtable()
        End If

        Dim eventHashCode As Object = GetEventDescriptorHashCode(e)

        Dim prop As PropertyDescriptor = DirectCast(_eventProperties(eventHashCode), PropertyDescriptor)

        If prop Is Nothing Then
            prop = New EventPropertyDescriptor(e, Me)
            _eventProperties(eventHashCode) = prop
        End If

        Return prop
    End Function

    'Displays the user code for this designer.  This will return true if the user
    'code could be displayed, or false otherwise.
    Private Function ShowCode() As Boolean Implements IEventBindingService.ShowCode
        Return ShowCode()
    End Function

    'Displays the user code for the designer.  This will return true if the user
    'code could be displayed, or false otherwise.
    Private Function ShowCode(ByVal lineNumber As Integer) As Boolean Implements IEventBindingService.ShowCode
        Return ShowCode(lineNumber)
    End Function

    'Displays the user code for the given event.  This will return true if the user
    'code could be displayed, or false otherwise.
    Private Function ShowCode(ByVal component As IComponent, ByVal e As EventDescriptor) As Boolean Implements IEventBindingService.ShowCode
        If component Is Nothing Then
            Throw New ArgumentNullException("component")
        End If

        If e Is Nothing Then
            Throw New ArgumentNullException("e")
        End If

        Dim prop As PropertyDescriptor = DirectCast(Me, IEventBindingService).GetEventProperty(e)

        Dim methodName As String = DirectCast(prop.GetValue(component), String)
        If methodName Is Nothing Then
            ' the event is not bound to a method.
            Return False
        End If

        Return ShowCode(component, e, methodName)
    End Function

    'This is an EventDescriptor cleverly wrapped in a PropertyDescriptor
    'of type String.  Note that we now handle subobjects by storing their
    'event information in their base component's site's dictionary.
    'Note also that when a value is set for this property we will code-gen
    'the event method.  If the property is set to a new value we will
    'remove the old event method ONLY if it is empty.
    Private Class EventPropertyDescriptor
        Inherits PropertyDescriptor
        Private _eventDesc As EventDescriptor
        Private _eventSvc As EventBindingService
        Private _converter As TypeConverter

        '''     Creates a new EventPropertyDescriptor.
        Friend Sub New(ByVal eventDesc As EventDescriptor, ByVal eventSvc As EventBindingService)
            MyBase.New(eventDesc, Nothing)
            _eventDesc = eventDesc
            _eventSvc = eventSvc
        End Sub

        'Indicates whether reset will change the value of the component.  If there
        'is a DefaultValueAttribute, then this will return true if getValue returns
        'something different than the default value.  If there is a reset method and
        'a shouldPersist method, this will return what shouldPersist returns.
        'If there is just a reset method, this always returns true.  If none of these
        'cases apply, this returns false.
        Public Overloads Overrides Function CanResetValue(ByVal component As Object) As Boolean
            Return GetValue(component) IsNot Nothing
        End Function

        'Retrieves the type of the component this PropertyDescriptor is bound to.
        Public Overloads Overrides ReadOnly Property ComponentType() As Type
            Get
                Return _eventDesc.ComponentType
            End Get
        End Property

        'Retrieves the type converter for this property.
        Public Overloads Overrides ReadOnly Property Converter() As TypeConverter
            Get
                If _converter Is Nothing Then
                    _converter = New EventConverter(_eventDesc)
                End If

                Return _converter
            End Get
        End Property

        'Retrieves the event descriptor we are representing.
        Friend ReadOnly Property [Event]() As EventDescriptor
            Get
                Return _eventDesc
            End Get
        End Property

        'Indicates whether this property is read only.
        Public Overloads Overrides ReadOnly Property IsReadOnly() As Boolean
            Get
                Return Attributes(GetType(ReadOnlyAttribute)).Equals(ReadOnlyAttribute.Yes)
            End Get
        End Property

        'Retrieves the type of the property.
        Public Overloads Overrides ReadOnly Property PropertyType() As Type
            Get
                Return _eventDesc.EventType
            End Get
        End Property

        'Retrieves the current value of the property on component,
        'invoking the getXXX method.  An exception in the getXXX
        'method will pass through.
        Public Overloads Overrides Function GetValue(ByVal component As Object) As Object

            If component Is Nothing Then
                Throw New ArgumentNullException("component")
            End If

            ' We must locate the sited component, because we store data on the dictionary
            ' service for the component.
            '
            Dim site As ISite = Nothing

            If TypeOf component Is IComponent Then
                site = DirectCast(component, IComponent).Site
            End If

            If site Is Nothing Then
                Dim rs As IReferenceService = TryCast(_eventSvc._provider.GetService(GetType(IReferenceService)), IReferenceService)
                If rs IsNot Nothing Then
                    Dim baseComponent As IComponent = rs.GetComponent(component)
                    If baseComponent IsNot Nothing Then
                        site = baseComponent.Site
                    End If
                End If
            End If

            If site Is Nothing Then
                Dim ex As Exception = New InvalidOperationException("There is no site for component" & component.ToString() & ".")
                Throw ex
            End If

            Dim ds As IDictionaryService = DirectCast(site.GetService(GetType(IDictionaryService)), IDictionaryService)
            If ds Is Nothing Then
                Dim ex As Exception = New InvalidOperationException("Cannot find IDictionaryService.")
                Throw ex
            End If

            Return DirectCast(ds.GetValue(New ReferenceEventClosure(component, Me)), String)
        End Function

        'Will reset the default value for this property on the component.  If
        'there was a default value passed in as a DefaultValueAttribute, that
        'value will be set as the value of the property on the component.  If
        'there was no default value passed in, a ResetXXX method will be looked
        'for.  If one is found, it will be invoked.  If one is not found, this
        'is a nop.
        Public Overloads Overrides Sub ResetValue(ByVal component As Object)
            SetValue(component, Nothing)
        End Sub

        'This will set value to be the new value of this property on the
        'component by invoking the setXXX method on the component.  If the
        'value specified is invalid, the component should throw an exception
        'which will be passed up.  The component designer should design the
        'property so that getXXX following a setXXX should return the value
        'passed in if no exception was thrown in the setXXX call.
        Public Overloads Overrides Sub SetValue(ByVal component As Object, ByVal value As Object)
            ' Argument, state checking.  Is it ok to set this event?
            '
            If IsReadOnly Then
                Dim ex As Exception = New InvalidOperationException("Tried to set a read only event.")
                Throw ex
            End If

            If value IsNot Nothing AndAlso Not (TypeOf value Is String) Then
                Dim ex As Exception = New ArgumentException("Cannot set to value " & value.ToString() & ".")
                Throw ex
            End If

            Dim name As String = DirectCast(value, String)
            If name IsNot Nothing AndAlso name.Length = 0 Then
                name = Nothing
            End If

            ' Obtain the site for the component.  Note that this can be a site
            ' to a parent component if we can get to the reference service.
            '
            Dim site As ISite = Nothing

            If TypeOf component Is IComponent Then
                site = DirectCast(component, IComponent).Site
            End If

            If site Is Nothing Then
                Dim rs As IReferenceService = TryCast(_eventSvc._provider.GetService(GetType(IReferenceService)), IReferenceService)
                If rs IsNot Nothing Then
                    Dim baseComponent As IComponent = rs.GetComponent(component)
                    If baseComponent IsNot Nothing Then
                        site = baseComponent.Site
                    End If
                End If
            End If

            If site Is Nothing Then
                Dim ex As Exception = New InvalidOperationException("There is no site for component " & component.ToString() & ".")
                Throw ex
            End If

            ' The dictionary service is where we store the actual event method name.
            '
            Dim ds As IDictionaryService = DirectCast(site.GetService(GetType(IDictionaryService)), IDictionaryService)
            If ds Is Nothing Then
                Dim ex As Exception = New InvalidOperationException("Cannot find IDictionaryService")
                Throw ex
            End If

            ' Get the old method name, ensure that they are different, and then continue.
            '
            Dim key As New ReferenceEventClosure(component, Me)
            Dim oldName As String = DirectCast(ds.GetValue(key), String)

            If Object.ReferenceEquals(oldName, name) Then
                Exit Sub
            End If

            If oldName IsNot Nothing AndAlso name IsNot Nothing AndAlso oldName.Equals(name) Then
                Exit Sub
            End If

            ' Before we continue our work, ensure that the name is
            ' actually valid.
            '
            If name IsNot Nothing Then
                _eventSvc.ValidateMethodName(name)
            End If

            ' Ok, the names are different.  Fire a changing event to make
            ' sure it's OK to perform the change.
            '
            Dim change As IComponentChangeService = DirectCast(site.GetService(GetType(IComponentChangeService)), IComponentChangeService)
            If change IsNot Nothing Then
                Try
                    change.OnComponentChanging(component, Me)
                Catch coEx As CheckoutException
                    If coEx Is CheckoutException.Canceled Then
                        Exit Sub
                    End If
                    Throw
                End Try
            End If

            ' Less chance of success of adding a new method name, so
            ' don't release the old name until we verify that adding
            ' the new one actually succeeded.
            '
            If name IsNot Nothing Then
                _eventSvc.UseMethod(component, _eventDesc, name)
            End If

            If oldName IsNot Nothing Then
                _eventSvc.FreeMethod(component, _eventDesc, oldName)
            End If

            ds.SetValue(key, name)

            If change IsNot Nothing Then
                change.OnComponentChanged(component, Me, oldName, name)
            End If

            OnValueChanged(component, EventArgs.Empty)
        End Sub

        'Indicates whether the value of this property needs to be persisted. In
        'other words, it indicates whether the state of the property is distinct
        'from when the component is first instantiated. If there is a default
        'value specified in this PropertyDescriptor, it will be compared against the
        'property's current value to determine this.  If there is't, the
        'shouldPersistXXX method is looked for and invoked if found.  If both
        'these routes fail, true will be returned.
        '
        'If this returns false, a tool should not persist this property's value.
        Public Overloads Overrides Function ShouldSerializeValue(ByVal component As Object) As Boolean
            Return CanResetValue(component)
        End Function

        'Implements a type converter for event objects.
        Private Class EventConverter
            Inherits TypeConverter

            Private _evt As EventDescriptor


            'Creates a new EventConverter.

            Friend Sub New(ByVal evt As EventDescriptor)
                _evt = evt
            End Sub


            'Determines if this converter can convert an object in the given source
            'type to the native type of the converter.
            Public Overloads Overrides Function CanConvertFrom(ByVal context As ITypeDescriptorContext, ByVal sourceType As Type) As Boolean
                If sourceType Is GetType(String) Then
                    Return True
                End If
                Return MyBase.CanConvertFrom(context, sourceType)
            End Function

            'Determines if this converter can convert an object to the given destination
            'type.
            Public Overloads Overrides Function CanConvertTo(ByVal context As ITypeDescriptorContext, ByVal destinationType As Type) As Boolean
                If destinationType Is GetType(String) Then
                    Return True
                End If
                Return MyBase.CanConvertTo(context, destinationType)
            End Function

            'Converts the given object to the converter's native type.
            Public Overloads Overrides Function ConvertFrom(ByVal context As ITypeDescriptorContext, ByVal culture As CultureInfo, ByVal value As Object) As Object
                If value Is Nothing Then
                    Return value
                End If
                If TypeOf value Is String Then
                    If DirectCast(value, String).Length = 0 Then
                        Return Nothing
                    End If
                    Return value
                End If
                Return MyBase.ConvertFrom(context, culture, value)
            End Function

            'Converts the given object to another type.  The most common types to convert
            'are to and from a string object.  The default implementation will make a call
            'to ToString on the object if the object is valid and if the destination
            'type is string.  If this cannot convert to the desitnation type, this will
            'throw a NotSupportedException.
            Public Overloads Overrides Function ConvertTo(ByVal context As ITypeDescriptorContext, ByVal culture As CultureInfo, ByVal value As Object, ByVal destinationType As Type) As Object
                If destinationType Is GetType(String) Then
                    If value Is Nothing Then
                        Return String.Empty
                    Else
                        Return value
                    End If
                End If
                Return MyBase.ConvertTo(context, culture, value, destinationType)
            End Function

            'Retrieves a collection containing a set of standard values
            'for the data type this validator is designed for.  This
            'will return null if the data type does not support a
            'standard set of values.
            Public Overloads Overrides Function GetStandardValues(ByVal context As ITypeDescriptorContext) As StandardValuesCollection
                ' We cannot cache this because it depends on the contents of the source file.
                '
                Dim eventMethods As String() = Nothing

                If context IsNot Nothing Then
                    Dim ebs As IEventBindingService = DirectCast(context.GetService(GetType(IEventBindingService)), IEventBindingService)
                    If ebs IsNot Nothing Then
                        Dim methods As ICollection = ebs.GetCompatibleMethods(_evt)
                        eventMethods = New String(methods.Count - 1) {}
                        Dim i As Integer = 0
                        For Each s As String In methods
                            eventMethods(System.Math.Max(System.Threading.Interlocked.Increment(i), i - 1)) = s
                        Next
                    End If
                End If

                Return New StandardValuesCollection(eventMethods)
            End Function

            'Determines if the list of standard values returned from
            'GetStandardValues is an exclusive list.  If the list
            'is exclusive, then no other values are valid, such as
            'in an enum data type.  If the list is not exclusive,
            'then there are other valid values besides the list of
            'standard values GetStandardValues provides.
            Public Overloads Overrides Function GetStandardValuesExclusive(ByVal context As ITypeDescriptorContext) As Boolean
                Return False
            End Function

            'Determines if this object supports a standard set of values
            ' that can be picked from a list.
            Public Overloads Overrides Function GetStandardValuesSupported(ByVal context As ITypeDescriptorContext) As Boolean
                Return True
            End Function
        End Class

        'This is a combination of a reference and a property, so that it can be used
        'as the key of a hashtable.  This is because we may have subobjects that share
        'the same property.
        Private Class ReferenceEventClosure
            Private reference As Object
            Private propertyDescriptor As EventPropertyDescriptor

            Public Sub New(ByVal reference As Object, ByVal prop As EventPropertyDescriptor)
                Me.reference = reference
                Me.propertyDescriptor = prop
            End Sub

            Public Overloads Overrides Function GetHashCode() As Integer
                Return reference.GetHashCode() * propertyDescriptor.GetHashCode()
            End Function

            Public Overloads Overrides Function Equals(ByVal otherClosure As [Object]) As Boolean
                If TypeOf otherClosure Is ReferenceEventClosure Then
                    Dim typedClosure As ReferenceEventClosure = DirectCast(otherClosure, ReferenceEventClosure)
                    Return (typedClosure.reference = reference AndAlso typedClosure.propertyDescriptor Is propertyDescriptor)
                End If
                Return False
            End Function
        End Class
    End Class
End Class

