'------------------------------------------------------------------------------
' <copyright from='1997' to='2002' company='Microsoft Corporation'>
'    Copyright (c) Microsoft Corporation. All Rights Reserved.
'
'    This source code is intended only as a supplement to Microsoft
'    Development Tools and/or on-line documentation.  See these other
'    materials for detailed information regarding Microsoft code samples.
'
' </copyright>
'------------------------------------------------------------------------------
Imports System
Imports System.Collections
Imports System.ComponentModel.Design.Serialization
Imports System.ComponentModel
Imports System.Reflection
Imports System.Text
Imports System.Diagnostics

Namespace FormDesigner
    '''     Our implementation of IDesignerSerializationManager. This is a private
    '''     implementation and we only have one.  So, it could have been implemented
    '''     as a private interface implementation on top of the loader class
    '''     itself.  I decided against this because there is a lot of
    '''     state associated with the serialization manager and it could get
    '''     confusing.
    Friend Class DesignerSerializationManager
        Implements IDesignerSerializationManager
        Private _loader As DesignLoader
        Private _resolveNameEventHandler As ResolveNameEventHandler
        Private _serializationCompleteEventHandler As EventHandler
        Private _designerSerializationProviders As ArrayList
        Private _instancesByName As Hashtable
        Private _namesByInstance As Hashtable
        Private _serializers As Hashtable
        Private _errorList As ArrayList
        Private _contextStack As ContextStack
        Private _propertyCollection As PropertyDescriptorCollection

        '''     Initializes the serialization manager.
        Friend Sub New(ByVal loader As DesignLoader)
            _loader = loader
        End Sub

        '''     This method is called by dependent loads to add additional
        '''     errors to the error list.  It is not called for the last
        '''     dependent load, nor is it called if IDesignerLoaderService
        '''     is not implemented.  If outside parties want to implement
        '''     IDesignerLoaderService, they may, but they will have
        '''     to provide their own storage for the dependent
        '''     error list.  We are just re-using _errorList here to be
        '''     efficient.
        Friend Sub AddErrors(ByVal errors As ICollection)
            If errors IsNot Nothing AndAlso errors.Count > 0 Then
                If _errorList Is Nothing Then
                    _errorList = New ArrayList()
                End If
                _errorList.AddRange(errors)
            End If
        End Sub

        '''     This starts the loading process.  Normally, everything 
        '''     should be cleared out when this is called.
        Friend Sub Initialize()
            If _errorList IsNot Nothing Then
                _errorList.Clear()
            End If
        End Sub

        '''     This ends the loading process.  This resets the state
        '''     of the serialization manager and merges the provided
        '''     error collection into the manager's own error list and
        '''     returns the merged list.
        Friend Function Terminate(ByVal errors As ICollection) As ICollection

            ' Let interested parties know that we're finished.
            '
            Try
                'RaiseEvent _serializationCompleteEventHandler(Me, EventArgs.Empty)
                RaiseEvent serializationComplete(Me, EventArgs.Empty)
            Catch
            End Try

            ' Merge the error list
            '
            If _errorList IsNot Nothing AndAlso _errorList.Count > 0 Then
                If errors IsNot Nothing AndAlso errors.Count > 0 Then
                    _errorList.AddRange(errors)
                End If
                errors = _errorList
            End If

            ' Now disolve our state.  The serialization manager
            ' should remain stateless.
            '
            _resolveNameEventHandler = Nothing
            _serializationCompleteEventHandler = Nothing
            _instancesByName = Nothing
            _namesByInstance = Nothing
            _serializers = Nothing
            _errorList = Nothing
            _contextStack = Nothing

            Return errors
        End Function

        '''     The Context property provides a user-defined storage area
        '''     implemented as a stack.  This storage area is a useful way
        '''     to provide communication across serializers, as serialization
        '''     is a generally hierarchial process.
        Private ReadOnly Property Context() As ContextStack Implements IDesignerSerializationManager.Context
            Get
                If _contextStack Is Nothing Then
                    _contextStack = New ContextStack()
                End If
                Return _contextStack
            End Get
        End Property

        '''     The Properties property provides a set of custom properties
        '''     the serialization manager may surface.  The set of properties
        '''     exposed here is defined by the implementor of 
        '''     IDesignerSerializationManager.  
        Private ReadOnly Property Properties() As PropertyDescriptorCollection Implements IDesignerSerializationManager.Properties
            Get
                If _propertyCollection Is Nothing Then
                    _propertyCollection = New PropertyDescriptorCollection(New PropertyDescriptor() {})
                End If
                Return _propertyCollection
            End Get
        End Property

        '''     ResolveName event.  This event
        '''     is raised when GetName is called, but the name is not found
        '''     in the serialization manager's name table.  It provides a 
        '''     way for a serializer to demand-create an object so the serializer
        '''     does not have to order object creation by dependency.  This
        '''     delegate is cleared immediately after serialization or deserialization
        '''     is complete.
        Private Custom Event ResolveName As ResolveNameEventHandler Implements IDesignerSerializationManager.ResolveName
            AddHandler(ByVal value As ResolveNameEventHandler)
                '_resolveNameEventHandler += value
                _resolveNameEventHandler = value
            End AddHandler
            RemoveHandler(ByVal value As ResolveNameEventHandler)
                '_resolveNameEventHandler -= value
                _resolveNameEventHandler = value
            End RemoveHandler
        End Event

        '''     This event is raised when serialization or deserialization
        '''     has been completed.  Generally, serialization code should
        '''     be written to be stateless.  Should some sort of state
        '''     be necessary to maintain, a serializer can listen to
        '''     this event to know when that state should be cleared.
        '''     An example of this is if a serializer needs to write
        '''     to another file, such as a resource file.  In this case
        '''     it would be inefficient to design the serializer
        '''     to close the file when finished because serialization of
        '''     an object graph generally requires several _serializers.
        '''     The resource file would be opened and closed many times.
        '''     Instead, the resource file could be accessed through
        '''     an object that listened to the SerializationComplete
        '''     event, and that object could close the resource file
        '''     at the end of serialization.
        Private Custom Event SerializationComplete As EventHandler Implements IDesignerSerializationManager.SerializationComplete
            AddHandler(ByVal value As EventHandler)
                '_serializationCompleteEventHandler += value
                _serializationCompleteEventHandler = value
            End AddHandler
            RemoveHandler(ByVal value As EventHandler)
                '_serializationCompleteEventHandler -= value
                _serializationCompleteEventHandler = value
            End RemoveHandler
        End Event

        '''     This method adds a custom serialization provider to the 
        '''     serialization manager.  A custom serialization provider will
        '''     get the opportunity to return a serializer for a data type
        '''     before the serialization manager looks in the type's
        '''     metadata.  
        Private Sub AddSerializationProvider(ByVal provider As IDesignerSerializationProvider) Implements IDesignerSerializationManager.AddSerializationProvider
            If _designerSerializationProviders Is Nothing Then
                _designerSerializationProviders = New ArrayList()
            End If
            _designerSerializationProviders.Add(provider)
        End Sub

        '''     Creates an instance of the given type and adds it to a collection
        '''     of named instances.  Objects that implement IComponent will be
        '''     added to the design time container if addToContainer is true.
        Private Function CreateInstance(ByVal type As Type, ByVal arguments As ICollection, ByVal name As String, ByVal addToContainer As Boolean) As Object Implements IDesignerSerializationManager.CreateInstance
            Dim argArray As Object() = Nothing

            If arguments IsNot Nothing AndAlso arguments.Count > 0 Then
                argArray = New Object(arguments.Count - 1) {}
                arguments.CopyTo(argArray, 0)
            End If

            Dim instance As Object = Nothing

            ' We do some special casing here.  If we are adding to the container, and if this type 
            ' is an IComponent, then we don't create the instance through Activator, we go
            ' through the loader host.  The reason for this is that if we went through activator,
            ' and if the object already specified a constructor that took an IContainer, our
            ' deserialization mechanism would equate the container to the designer host.  This
            ' is the correct thing to do, but it has the side effect of adding the component
            ' to the designer host twice -- once with a default name, and a second time with
            ' the name we provide.  This equates to a component rename, which isn't cheap, 
            ' so we don't want to do it when we load each and every component.
            '
            If addToContainer AndAlso GetType(IComponent).IsAssignableFrom(type) Then
                Dim host As IDesignerLoaderHost = _loader.LoaderHost
                If host IsNot Nothing Then
                    If name Is Nothing Then
                        instance = host.CreateComponent(type)
                    Else
                        instance = host.CreateComponent(type, name)
                    End If
                End If
            End If

            If instance Is Nothing Then

                ' If we have a name but the object wasn't a component
                ' that was added to the design container, save the
                ' name/value relationship in our own nametable.
                '
                If name IsNot Nothing AndAlso _instancesByName IsNot Nothing Then
                    If _instancesByName.ContainsKey(name) Then
                        Dim ex As Exception = New InvalidOperationException("Duplicate component declaration for " & name & ".")
                        Throw ex
                    End If
                End If

                Try
                    instance = Activator.CreateInstance(type, BindingFlags.Instance Or BindingFlags.[Public] Or BindingFlags.CreateInstance, Nothing, argArray, Nothing)
                Catch generatedExceptionName As MissingMethodException
                    Dim argTypes As New StringBuilder()
                    For Each o As Object In argArray
                        If argTypes.Length > 0 Then
                            argTypes.Append(", ")
                        End If

                        If o IsNot Nothing Then
                            argTypes.Append(o.[GetType]().Name)
                        Else
                            argTypes.Append("null")

                        End If
                    Next

                    Dim ex As Exception = New InvalidOperationException(("No matching constructor for " & type.FullName & "(") + argTypes.ToString() & ")")
                    Throw ex
                End Try

                ' If we have a name but the object wasn't a component
                ' that was added to the design container, save the
                ' name/value relationship in our own nametable.
                '
                If name IsNot Nothing Then
                    If _instancesByName Is Nothing Then
                        _instancesByName = New Hashtable()
                        _namesByInstance = New Hashtable()
                    End If

                    _instancesByName(name) = instance
                    _namesByInstance(instance) = name
                End If
            End If

            Return instance
        End Function

        '''     Retrieves an instance of a created object of the given name, or
        '''     null if that object does not exist.
        Private Function GetInstance(ByVal name As String) As Object Implements IDesignerSerializationManager.GetInstance
            Dim instance As Object = Nothing

            If name Is Nothing Then
                Throw New ArgumentNullException("name")
            End If

            ' Check our local nametable first
            '
            If _instancesByName IsNot Nothing Then
                instance = _instancesByName(name)
            End If

            If instance Is Nothing AndAlso _loader.LoaderHost IsNot Nothing Then
                instance = _loader.LoaderHost.Container.Components(name)
            End If

            If instance Is Nothing AndAlso _resolveNameEventHandler IsNot Nothing Then
                Dim e As New ResolveNameEventArgs(name)
                _resolveNameEventHandler(Me, e)
                instance = e.Value
            End If

            Return instance
        End Function

        '''     Retrieves a name for the specified object, or null if the object
        '''     has no name.
        Private Function GetName(ByVal value As Object) As String Implements IDesignerSerializationManager.GetName
            Dim name As String = Nothing

            If value Is Nothing Then
                Throw New ArgumentNullException("value")
            End If

            ' Check our local nametable first
            '
            If _namesByInstance IsNot Nothing Then
                name = DirectCast(_namesByInstance(value), String)
            End If

            If name Is Nothing AndAlso TypeOf value Is IComponent Then
                Dim site As ISite = DirectCast(value, IComponent).Site
                If site IsNot Nothing Then
                    name = site.Name
                End If
            End If
            Return name
        End Function

        '''     Retrieves a serializer of the requested type for the given
        '''     object type.
        Private Function GetSerializer(ByVal objectType As Type, ByVal serializerType As Type) As Object Implements IDesignerSerializationManager.GetSerializer
            Dim serializer As Object = Nothing

            If objectType IsNot Nothing Then

                If _serializers IsNot Nothing Then
                    ' I don't double hash here.  It will be a very rare day where
                    ' multiple types of serializers will be used in the same scheme.
                    ' We still handle it, but we just don't cache.
                    '
                    serializer = _serializers(objectType)
                    If serializer IsNot Nothing AndAlso Not serializerType.IsAssignableFrom(serializer.[GetType]()) Then
                        serializer = Nothing
                    End If
                End If

                ' Now actually look in the type's metadata.
                '
                Dim host As IDesignerLoaderHost = _loader.LoaderHost
                If serializer Is Nothing AndAlso host IsNot Nothing Then
                    Dim attributes As AttributeCollection = TypeDescriptor.GetAttributes(objectType)
                    For Each attr As Attribute In attributes
                        If TypeOf attr Is DesignerSerializerAttribute Then
                            Dim da As DesignerSerializerAttribute = DirectCast(attr, DesignerSerializerAttribute)
                            Dim typeName As String = da.SerializerBaseTypeName

                            ' This serializer must support a CodeDomSerializer or we're not interested.
                            '
                            If typeName IsNot Nothing AndAlso host.[GetType](typeName) Is serializerType AndAlso da.SerializerTypeName IsNot Nothing AndAlso da.SerializerTypeName.Length > 0 Then
                                Dim type As Type = host.[GetType](da.SerializerTypeName)
                                Debug.Assert(type IsNot Nothing, ("Type " & objectType.FullName & " has a serializer that we couldn't bind to: ") + da.SerializerTypeName)
                                If type IsNot Nothing Then
                                    serializer = Activator.CreateInstance(type, BindingFlags.Instance Or BindingFlags.[Public] Or BindingFlags.NonPublic Or BindingFlags.CreateInstance, Nothing, Nothing, Nothing)
                                    Exit For
                                End If
                            End If
                        End If
                    Next

                    ' And stash this little guy for later.
                    '
                    If serializer IsNot Nothing Then
                        If _serializers Is Nothing Then
                            _serializers = New Hashtable()
                        End If
                        _serializers(objectType) = serializer
                    End If
                End If
            End If

            ' Designer serialization providers can override our metadata discovery.
            ' We loop until we reach steady state.  This breaks order dependencies
            ' by allowing all providers a chance to party on each other's serializers.
            '
            If _designerSerializationProviders IsNot Nothing Then
                Dim continueLoop As Boolean = True
                Dim i As Integer = 0
                While continueLoop AndAlso i < _designerSerializationProviders.Count

                    continueLoop = False

                    For Each provider As IDesignerSerializationProvider In _designerSerializationProviders
                        Dim newSerializer As Object = provider.GetSerializer(Me, serializer, objectType, serializerType)
                        If newSerializer IsNot Nothing Then
                            continueLoop = (serializer <> newSerializer)
                            serializer = newSerializer
                        End If
                    Next
                    i += 1
                End While
            End If

            Return serializer
        End Function

        '''     Retrieves a type of the given name.
        Private Function [GetType](ByVal typeName As String) As Type Implements IDesignerSerializationManager.[GetType]
            Dim t As Type = Nothing

            If _loader.LoaderHost IsNot Nothing Then
                While t Is Nothing
                    t = _loader.LoaderHost.[GetType](typeName)

                    If t Is Nothing AndAlso typeName IsNot Nothing AndAlso typeName.Length > 0 Then
                        Dim dotIndex As Integer = typeName.LastIndexOf("."c)
                        If dotIndex = -1 OrElse dotIndex = typeName.Length - 1 Then
                            Exit While
                        End If

                        typeName = (typeName.Substring(0, dotIndex) & "+") + typeName.Substring(dotIndex + 1, typeName.Length - dotIndex - 1)
                    End If
                End While
            End If

            Return t
        End Function

        '''     Removes a previously added serialization provider.
        Private Sub RemoveSerializationProvider(ByVal provider As IDesignerSerializationProvider) Implements IDesignerSerializationManager.RemoveSerializationProvider
            If _designerSerializationProviders IsNot Nothing Then
                _designerSerializationProviders.Remove(provider)
            End If
        End Sub

        '''     Reports a non-fatal error in serialization.  The serialization
        '''     manager may implement a logging scheme to alert the caller
        '''     to all non-fatal errors at once.  If it doesn't, it should
        '''     immediately throw in this method, which should abort
        '''     serialization.  
        '''     Serialization may continue after calling this function.
        Private Sub ReportError(ByVal errorInformation As Object) Implements IDesignerSerializationManager.ReportError
            If errorInformation IsNot Nothing Then
                If _errorList Is Nothing Then
                    _errorList = New ArrayList()
                End If

                _errorList.Add(errorInformation)
            End If
        End Sub

        '''     Provides a way to set the name of an existing object.
        '''     This is useful when it is necessary to create an 
        '''     instance of an object without going through CreateInstance.
        '''     An exception will be thrown if you try to rename an existing
        '''     object or if you try to give a new object a name that
        '''     is already taken.
        Private Sub SetName(ByVal instance As Object, ByVal name As String) Implements IDesignerSerializationManager.SetName

            If instance Is Nothing Then
                Throw New ArgumentNullException("instance")
            End If

            If name Is Nothing Then
                Throw New ArgumentNullException("name")
            End If

            If _instancesByName Is Nothing Then
                _instancesByName = New Hashtable()
                _namesByInstance = New Hashtable()
            End If

            If _instancesByName(name) IsNot Nothing Then
                Dim ex As Exception = New ArgumentException("Designer Loader name " & name & " in use.")
                Throw ex
            End If

            If _namesByInstance(instance) IsNot Nothing Then
                Dim ex As Exception = New ArgumentException("Designer loader object has name " & name & ".")
                Throw ex
            End If

            _instancesByName(name) = instance
            _namesByInstance(instance) = name
        End Sub

        '''     Retrieves the requested service.
        Private Function GetService(ByVal serviceType As Type) As Object Implements IServiceProvider.GetService
            Return _loader.GetService(serviceType)
        End Function
    End Class
End Namespace
