'==============================================================================
' Basiert auf der Arbeit von Tim Dawson, www.divil.co.uk
'==============================================================================
Imports System
Imports System.Collections
Imports System.ComponentModel
Imports System.ComponentModel.Design
Imports System.ComponentModel.Design.Serialization


<ProvideProperty("Name", GetType(IComponent))> _
Friend Class DesignerHost
    Implements IDesignerHost
    Implements IContainer
    Implements IComponentChangeService
    Implements IExtenderProvider
    Implements ITypeDescriptorFilterService
    Implements IExtenderListService
    Implements IExtenderProviderService

    ' Transactions
    Private _transactions As Stack = Nothing

    ' Services
    Private _parent As IServiceContainer = Nothing

    ' Container
    Private _components As Hashtable = Nothing
    Private _designers As Hashtable = Nothing
    Private _extenderProviders As ArrayList = Nothing
    Private _rootComponent As IComponent = Nothing

    Public Sub New(ByVal parent As IServiceContainer)
        ' Keep the parent reference around for re-use
        _parent = parent

        Dim myComparer As IEqualityComparer = Nothing

        ' Initialise container helpers
        _components = New Hashtable(myComparer)
        _designers = New Hashtable()

        ' Initialise transaction stack
        _transactions = New Stack()

        ' Add our own services
        _parent.AddService(GetType(IDesignerHost), Me)
        _parent.AddService(GetType(IContainer), Me)
        _parent.AddService(GetType(IComponentChangeService), Me)
        _parent.AddService(GetType(ITypeDescriptorFilterService), Me)

        ' Add extender services
        _extenderProviders = New ArrayList()
        _parent.AddService(GetType(IExtenderListService), Me)
        _parent.AddService(GetType(IExtenderProviderService), Me)

        AddExtenderProvider(Me)

        ' Add selection service
        parent.AddService(GetType(ISelectionService), New SelectionService(Me))

    End Sub

#Region " IServiceContainer Implementation "
    Public Function GetService(ByVal serviceType As System.Type) As Object Implements IServiceContainer.GetService
        Return _parent.GetService(serviceType)
    End Function

    Public Sub AddService(ByVal serviceType As System.Type, ByVal callback As System.ComponentModel.Design.ServiceCreatorCallback, ByVal promote As Boolean) Implements IServiceContainer.AddService
        _parent.AddService(serviceType, callback, promote)
    End Sub

    Public Sub AddService(ByVal serviceType As System.Type, ByVal callback As System.ComponentModel.Design.ServiceCreatorCallback) Implements IServiceContainer.AddService
        _parent.AddService(serviceType, callback)
    End Sub

    Public Sub AddService(ByVal serviceType As System.Type, ByVal serviceInstance As Object, ByVal promote As Boolean) Implements IServiceContainer.AddService
        _parent.AddService(serviceType, serviceInstance, promote)
    End Sub

    Public Sub AddService(ByVal serviceType As System.Type, ByVal serviceInstance As Object) Implements IServiceContainer.AddService
        _parent.AddService(serviceType, serviceInstance)
    End Sub

    Public Sub RemoveService(ByVal serviceType As System.Type, ByVal promote As Boolean) Implements IServiceContainer.RemoveService
        _parent.RemoveService(serviceType, promote)
    End Sub

    Public Sub RemoveService(ByVal serviceType As System.Type) Implements IServiceContainer.RemoveService
        _parent.RemoveService(serviceType)
    End Sub
#End Region

#Region " IDesignerHost Implementation "
    Public Sub Activate() Implements IDesignerHost.Activate
        Dim s As ISelectionService = CType(GetService(GetType(ISelectionService)), ISelectionService)

        ' Simply set the root component as the primary selection
        If Not (s Is Nothing) Then
            Dim o() As Object = {_rootComponent}
            s.SetSelectedComponents(o)

            RaiseEvent Activated(Me, EventArgs.Empty)
        End If

    End Sub

    Public Function CreateComponent(ByVal componentClass As System.Type, ByVal name As String) As System.ComponentModel.IComponent Implements IDesignerHost.CreateComponent
        Dim Component As IComponent = Nothing

        ' Create instance
        Component = CType(Activator.CreateInstance(componentClass), IComponent)

        ' Add to design container
        Add(Component, name)

        Return Component
    End Function

    Public Function CreateComponent(ByVal componentClass As System.Type) As System.ComponentModel.IComponent Implements IDesignerHost.CreateComponent
        Return CreateComponent(componentClass, Nothing)
    End Function

    Public Function CreateTransaction(ByVal description As String) As System.ComponentModel.Design.DesignerTransaction Implements IDesignerHost.CreateTransaction
        Dim transaction As DesignerTransaction = Nothing

        ' Raise event if me is the first transaction in a chain
        If (_transactions.Count = 0) Then
            RaiseEvent TransactionOpening(Me, EventArgs.Empty)
        End If

        ' Create transaction
        If (description Is Nothing) Then
            transaction = New MegaDesignerTransaction(Me)
        Else
            transaction = New MegaDesignerTransaction(Me, description)
        End If

        _transactions.Push(transaction)

        ' Let people know a transaction has opened
        RaiseEvent TransactionOpened(Me, EventArgs.Empty)

        Return transaction
    End Function

    Friend Sub OnTransactionClosing(ByVal commit As Boolean)
        RaiseEvent TransactionClosing(Me, New DesignerTransactionCloseEventArgs(commit, commit))
    End Sub

    Friend Sub OnTransactionClosed(ByVal commit As Boolean)
        RaiseEvent TransactionClosed(Me, New DesignerTransactionCloseEventArgs(commit, commit))

        _transactions.Pop()
    End Sub

    Public Function CreateTransaction() As System.ComponentModel.Design.DesignerTransaction Implements IDesignerHost.CreateTransaction
        Return CreateTransaction(Nothing)
    End Function

    Public Sub DestroyComponent(ByVal component As System.ComponentModel.IComponent) Implements IDesignerHost.DestroyComponent
        Dim t As DesignerTransaction = Nothing

        ' Create transaction
        t = CreateTransaction("Destroy Component")

        ' Destroy component
        If (component.Site.Container Is Me) Then
            OnComponentChanging(component, Nothing)
            Remove(component)
            component.Dispose()
            OnComponentChanged(component, Nothing, Nothing, Nothing)
        End If

        ' Commit transaction
        t.Commit()
    End Sub

    Public Function GetDesigner(ByVal component As System.ComponentModel.IComponent) As System.ComponentModel.Design.IDesigner Implements IDesignerHost.GetDesigner
        If (component Is Nothing) Then
            Return Nothing
        End If

        Return CType(_designers(component), IDesigner)
    End Function

    Public Function GetTheType(ByVal typeName As String) As System.Type Implements IDesignerHost.GetType
        Dim typeResolver As ITypeResolutionService = CType(GetService(GetType(ITypeResolutionService)), ITypeResolutionService)

        If (typeResolver Is Nothing) Then
            Return Type.GetType(typeName)
        Else
            Return typeResolver.GetType(typeName)
        End If
    End Function

    Public ReadOnly Property Container() As System.ComponentModel.IContainer Implements IDesignerHost.Container
        Get
            Return Me
        End Get
    End Property

    Public ReadOnly Property InTransaction() As Boolean Implements IDesignerHost.InTransaction
        Get
            Return (_transactions.Count > 0)
        End Get
    End Property

    Public ReadOnly Property Loading() As Boolean Implements IDesignerHost.Loading
        Get
            Return False
        End Get
    End Property

    Public ReadOnly Property RootComponent() As System.ComponentModel.IComponent Implements IDesignerHost.RootComponent
        Get
            Return _rootComponent
        End Get
    End Property

    Public ReadOnly Property RootComponentClassName() As String Implements IDesignerHost.RootComponentClassName
        Get
            Return CType(_rootComponent, Object).GetType().Name
        End Get
    End Property

    Public ReadOnly Property TransactionDescription() As String Implements IDesignerHost.TransactionDescription
        Get
            If (InTransaction) Then
                Dim t As DesignerTransaction = CType(_transactions.Peek(), DesignerTransaction)
                Return t.Description
            Else
                Return Nothing
            End If
        End Get
    End Property

    Public Event Activated As System.EventHandler Implements IDesignerHost.Activated
    Public Event Deactivated As System.EventHandler Implements IDesignerHost.Deactivated
    Public Event LoadComplete As System.EventHandler Implements IDesignerHost.LoadComplete
    Public Event TransactionClosed As System.ComponentModel.Design.DesignerTransactionCloseEventHandler Implements IDesignerHost.TransactionClosed
    Public Event TransactionClosing As System.ComponentModel.Design.DesignerTransactionCloseEventHandler Implements IDesignerHost.TransactionClosing
    Public Event TransactionOpened As System.EventHandler Implements IDesignerHost.TransactionOpened
    Public Event TransactionOpening As System.EventHandler Implements IDesignerHost.TransactionOpening
#End Region

#Region " IContainer Implementation "
    Friend Function ContainsName(ByVal name As String) As Boolean
        Return (_components.Contains(name))
    End Function

    Public Sub Add(ByVal component As System.ComponentModel.IComponent, ByVal name As String) Implements IContainer.Add
        Dim designer As IDesigner = Nothing
        Dim site As DesignSite = Nothing

        ' Check we're not trying to add a null component
        If (component Is Nothing) Then
            Throw New ArgumentNullException("Eine nicht initialiserte Komponente kann dem Container nicht hizugefgt werden.") '"Cannot add a null component to the container."
        End If

        ' Remove this component from its existing container, if applicable
        If Not component.Site Is Nothing Then
            If component.Site.Container Is Me Then
                component.Site.Container.Remove(component)
            End If
        End If

        ' Make sure we have a name for the component
        If (name Is Nothing) Then
            Dim nameService As INameCreationService = CType(GetService(GetType(INameCreationService)), INameCreationService)
            name = nameService.CreateName(Me, CType(component, Object).GetType())
        End If

        ' Make sure there isn't already a component with this name in the container
        If (ContainsName(name)) Then
            Throw New ArgumentException("Eine Komponente mit diesem Namen existiert bereits.") '"A component with this name already exists in the container."
        End If

        ' Give the new component a site
        site = New DesignSite(Me, name)
        site.SetComponent(component)
        component.Site = site

        ' Let everyone know there's a component being added
        RaiseEvent ComponentAdding(Me, New ComponentEventArgs(component))

        ' Get the designer for this component
        If (_components.Count = 0) Then
            ' This is the first component being added and therefore must offer a root designer
            designer = TypeDescriptor.CreateDesigner(component, GetType(IRootDesigner))
            _rootComponent = component
        Else
            designer = TypeDescriptor.CreateDesigner(component, GetType(IDesigner))
        End If

        ' If we got a designer, initialize it
        If (Not designer Is Nothing) Then
            designer.Initialize(component)
            _designers(component) = designer
        Else
            ' This should never happen
            component.Site = Nothing
            Throw New InvalidOperationException("Fehler beim Designer ffnen.") '"Failed to get designer for this component."
        End If

        ' Add to our list of extenderproviders if necessary
        If (TypeOf component Is IExtenderProvider) Then
            Dim e As IExtenderProviderService = CType(GetService(GetType(IExtenderProviderService)), IExtenderProviderService)
            e.AddExtenderProvider(CType(component, IExtenderProvider))
        End If

        ' Finally we're able to add the component
        _components.Add(component.Site.Name, component)

        RaiseEvent ComponentAdded(Me, New ComponentEventArgs(component))
    End Sub

    Public Sub Add(ByVal component As System.ComponentModel.IComponent) Implements IContainer.Add
        Add(component, Nothing)
    End Sub

    Public Sub Remove(ByVal component As System.ComponentModel.IComponent) Implements IContainer.Remove
        Dim site As ISite = component.Site
        Dim designer As IDesigner = Nothing

        ' Make sure component isn't null
        If (component Is Nothing) Then
            Return
        End If

        ' Make sure component is sited here
        If (component.Site Is Nothing Or Not (component.Site.Container Is Me)) Then
            Return
        End If

        ' Let the nice people know the component is being removed
        RaiseEvent ComponentRemoving(Me, New ComponentEventArgs(component))

        ' Remove extender provider (if any)
        If (TypeOf component Is IExtenderProvider) Then
            Dim e As IExtenderProviderService = CType(GetService(GetType(IExtenderProviderService)), IExtenderProviderService)
            e.RemoveExtenderProvider(CType(component, IExtenderProvider))
        End If

        ' Remove the component and dispose of its designer
        _components.Remove(site.Name)
        designer = CType(_designers(component), IDesigner)
        If Not (designer Is Nothing) Then
            designer.Dispose()
            _designers.Remove(component)
        End If

        ' Let the nice people know the component has been removed
        RaiseEvent ComponentRemoved(Me, New ComponentEventArgs(component))

        ' Kill the component's site
        component.Site = Nothing
    End Sub

    Public ReadOnly Property Components() As System.ComponentModel.ComponentCollection Implements IContainer.Components
        Get
            Dim c() As IComponent = {}

            ' If there are no components (shouldn't happen)
            If _components.Count = 0 Then
                Return New ComponentCollection(c)
            End If

            ' Compile list and return it
            ReDim c(_components.Count)

            _components.Values.CopyTo(c, 0)

            Return New ComponentCollection(c)
        End Get
    End Property

    Public Sub Dispose() Implements IContainer.Dispose
        Dim key As String

        Dim keys As ICollection = _components.Keys

        ' Remove and dispose of all components in our collection
        For Each key In keys
            Dim component As IComponent = CType(_components(key), IComponent)
            _components.Remove(component)
            component.Dispose()
        Next

        _components.Clear()
    End Sub
#End Region

#Region " IComponentChangeService Implementation "

    Public Sub OnComponentChanged(ByVal component As Object, ByVal member As System.ComponentModel.MemberDescriptor, ByVal oldValue As Object, ByVal newValue As Object) Implements IComponentChangeService.OnComponentChanged
        RaiseEvent ComponentChanged(Me, New ComponentChangedEventArgs(component, member, oldValue, newValue))
    End Sub

    Public Sub OnComponentChanging(ByVal component As Object, ByVal member As System.ComponentModel.MemberDescriptor) Implements IComponentChangeService.OnComponentChanging
        RaiseEvent ComponentChanging(Me, New ComponentChangingEventArgs(component, member))
    End Sub

    Friend Sub OnComponentRename(ByVal component As Object, ByVal oldName As String, ByVal newName As String)
        RaiseEvent ComponentRename(Me, New ComponentRenameEventArgs(component, oldName, newName))
    End Sub

    Public Event ComponentAdded As System.ComponentModel.Design.ComponentEventHandler Implements IComponentChangeService.ComponentAdded
    Public Event ComponentAdding As System.ComponentModel.Design.ComponentEventHandler Implements IComponentChangeService.ComponentAdding
    Public Event ComponentChanged As System.ComponentModel.Design.ComponentChangedEventHandler Implements IComponentChangeService.ComponentChanged
    Public Event ComponentChanging As System.ComponentModel.Design.ComponentChangingEventHandler Implements IComponentChangeService.ComponentChanging
    Public Event ComponentRemoved As System.ComponentModel.Design.ComponentEventHandler Implements IComponentChangeService.ComponentRemoved
    Public Event ComponentRemoving As System.ComponentModel.Design.ComponentEventHandler Implements IComponentChangeService.ComponentRemoving
    Public Event ComponentRename As System.ComponentModel.Design.ComponentRenameEventHandler Implements IComponentChangeService.ComponentRename
#End Region

#Region " IExtenderProvider Implementation "
    Public Function CanExtend(ByVal extendee As Object) As Boolean Implements IExtenderProvider.CanExtend
        Return (TypeOf extendee Is IComponent)
    End Function

    <DesignOnly(True), Category("Design"), Browsable(True), ParenthesizePropertyName(True), Description("The variable used to refer to this component in source code.")> Public Function GetName(ByVal component As IComponent) As String
        ' Make sure component is sited
        If (component.Site Is Nothing) Then
            Throw New InvalidOperationException("Komponente ist nicht im Design-Modus.") '"Component is not sited."
        End If

        Return component.Site.Name
    End Function

    Public Sub SetName(ByVal component As IComponent, ByVal name As String)
        ' Make sure component is sited
        If (component.Site Is Nothing) Then
            Throw New InvalidOperationException("Komponente ist nicht im Design-Modus.") '"Component is not sited."
        End If

        component.Site.Name = name
    End Sub
#End Region

#Region " ITypeDescriptorFilterService Implementation "
    Public Function FilterAttributes(ByVal component As IComponent, ByVal attributes As IDictionary) As Boolean Implements ITypeDescriptorFilterService.FilterAttributes
        Dim designer As IDesigner = GetDesigner(component)
        If (TypeOf designer Is IDesignerFilter) Then
            CType(designer, IDesignerFilter).PreFilterAttributes(attributes)
            CType(designer, IDesignerFilter).PostFilterAttributes(attributes)
        End If

        Return (designer Is Nothing) = False
    End Function

    Public Function FilterEvents(ByVal component As IComponent, ByVal events As IDictionary) As Boolean Implements ITypeDescriptorFilterService.FilterEvents
        Dim designer As IDesigner = GetDesigner(component)
        If (TypeOf designer Is IDesignerFilter) Then
            CType(designer, IDesignerFilter).PreFilterEvents(events)
            CType(designer, IDesignerFilter).PostFilterEvents(events)
        End If

        Return (designer Is Nothing) = False
    End Function

    Public Function FilterProperties(ByVal component As IComponent, ByVal properties As IDictionary) As Boolean Implements ITypeDescriptorFilterService.FilterProperties
        Dim designer As IDesigner = GetDesigner(component)
        If (TypeOf designer Is IDesignerFilter) Then
            CType(designer, IDesignerFilter).PreFilterProperties(properties)
            CType(designer, IDesignerFilter).PostFilterProperties(properties)
        End If

        Return (designer Is Nothing) = False
    End Function
#End Region

#Region " IExtenderListService Implementation "
    Public Function GetExtenderProviders() As System.ComponentModel.IExtenderProvider() Implements IExtenderListService.GetExtenderProviders
        ' Copy our collection into an array to return
        Dim e() As IExtenderProvider
        ReDim e(_extenderProviders.Count - 1)
        _extenderProviders.CopyTo(e, 0)

        Return e
    End Function
#End Region

#Region " IExtenderProviderService Implementation "
    Public Sub AddExtenderProvider(ByVal provider As System.ComponentModel.IExtenderProvider) Implements IExtenderProviderService.AddExtenderProvider
        If (Not _extenderProviders.Contains(provider)) Then
            _extenderProviders.Add(provider)
        End If
    End Sub

    Public Sub RemoveExtenderProvider(ByVal provider As System.ComponentModel.IExtenderProvider) Implements IExtenderProviderService.RemoveExtenderProvider
        If (_extenderProviders.Contains(provider)) Then
            _extenderProviders.Remove(provider)
        End If
    End Sub
#End Region
End Class
