'------------------------------------------------------------------------------
' <copyright from='1997' to='2002' company='Microsoft Corporation'>
'    Copyright (c) Microsoft Corporation. All Rights Reserved.
'
'    This source code is intended only as a supplement to Microsoft
'    Development Tools and/or on-line documentation.  See these other
'    materials for detailed information regarding Microsoft code samples.
'
' </copyright>
'------------------------------------------------------------------------------
Imports System
Imports System.ComponentModel
Imports System.ComponentModel.Design
Imports System.ComponentModel.Design.Serialization
Imports System.Collections
Imports System.Diagnostics
Imports System.Globalization
Imports System.IO
Imports System.Reflection
Imports System.Runtime.Serialization.Formatters.Binary
Imports System.Text
Imports System.Windows.Forms
Imports System.Xml
Imports System.CodeDom.Compiler
Imports System.CodeDom
Imports Microsoft.CSharp
Imports Microsoft.VisualBasic
Namespace FormDesigner

    ''' This is a designer loader that is based on XML.  We use reflection
    ''' to write out values into an XML document.  The techniques used in this
    ''' designer loader to discover, via reflection, the properties and
    ''' objects that need to be saved or loaded can be applied to any
    ''' persistence format.
    ''' 
    ''' The XML format we use here is not terribly user-friendly, but
    ''' is fairly straightforward.  It handles the vast majority of
    ''' persistence requirements including collections, instance descriptors,
    ''' and binary data.
    ''' 
    ''' In addition to maintaining the buffer in the form of an XmlDocument,
    ''' we also maintain it in a CodeCompileUnit. We use this DOM to generate
    ''' C# and VB code, as well as to compile the buffer into an executable.
    Public Class DesignLoader
        Inherits DesignerLoader
        Private dirty As Boolean
        Private unsaved As Boolean
        Private fileName As String
        Private executable As String
        Private host As IDesignerLoaderHost
        Private xmlDocument As XmlDocument
        Private codeCompileUnit As CodeCompileUnit
        Private run As Process

        Private Shared ReadOnly propertyAttributes As Attribute() = New Attribute() {DesignOnlyAttribute.No}

        ''' Empty constructor simply creates a new form.
        Public Sub New()
        End Sub

        ''' This constructor takes a file name.  This file
        ''' should exist on disk and consist of XML that
        ''' can be read by a data set.
        Public Sub New(ByVal fileName As String)
            If fileName Is Nothing Then
                Throw New ArgumentNullException("fileName")
            End If
            Me.fileName = fileName
        End Sub

        ' Called by the host when we load a document.
        Public Overloads Overrides Sub BeginLoad(ByVal host As IDesignerLoaderHost)
            If host Is Nothing Then
                Throw New ArgumentNullException("DesignLoader.BeginLoad: Invalid designerLoaderHost.")
            End If

            Me.host = host

            ' The loader will put error messages in here.
            '
            Dim errors As New ArrayList()
            Dim successful As Boolean = True
            Dim baseClassName As String

            ' The loader is responsible for providing certain services to the host.
            '
            host.AddService(GetType(IDesignerSerializationManager), New DesignerSerializationManager(Me))
            host.AddService(GetType(IEventBindingService), New EventBindingService(host))
            host.AddService(GetType(ITypeResolutionService), New TypeResolutionService())
            host.AddService(GetType(CodeDomProvider), New CSharpCodeProvider())
            host.AddService(GetType(IResourceService), New ResourceService(host))

            ' If no filename was passed in, just create a form and be done with it.  If a file name
            ' was passed, read it.
            '
            If fileName Is Nothing Then
                baseClassName = host.CreateComponent(GetType(System.Windows.Forms.Form)).Site.Name
            Else
                baseClassName = ReadFile(fileName, errors, xmlDocument)
            End If

            ' Now that we are done with the load work, we need to begin to listen to events.
            ' Listening to event notifications is how a designer "Loader" can also be used
            ' to save data.  If we wanted to integrate this loader with source code control,
            ' we would listen to the "ing" events as well as the "ed" events.
            '
            Dim cs As IComponentChangeService = TryCast(host.GetService(GetType(IComponentChangeService)), IComponentChangeService)
            If cs IsNot Nothing Then
                AddHandler cs.ComponentChanged, AddressOf OnComponentChanged
                AddHandler cs.ComponentAdded, AddressOf OnComponentAddedRemoved
                AddHandler cs.ComponentRemoved, AddressOf OnComponentAddedRemoved
            End If

            ' Let the host know we are done loading.
            host.EndLoad(baseClassName, successful, errors)

            ' We've just loaded a document, so you can bet we need to flush changes.
            dirty = True
            unsaved = False
        End Sub

        Public Overloads Overrides Sub Dispose()
            ' Always remove attached event handlers in Dispose.
            Dim cs As IComponentChangeService = TryCast(host.GetService(GetType(IComponentChangeService)), IComponentChangeService)
            If cs IsNot Nothing Then
                RemoveHandler cs.ComponentChanged, AddressOf OnComponentChanged
                RemoveHandler cs.ComponentAdded, AddressOf OnComponentAddedRemoved
                RemoveHandler cs.ComponentRemoved, AddressOf OnComponentAddedRemoved
            End If
        End Sub

        ''' This method is called by the designer host whenever it wants the
        ''' designer loader to flush any pending changes.  Flushing changes
        ''' does not mean the same thing as saving to disk.  For example,
        ''' In Visual Studio, flushing changes causes new code to be generated
        ''' and inserted into the text editing window.  The user can edit
        ''' the new code in the editing window, but nothing has been saved
        ''' to disk.  This sample adheres to this separation between flushing
        ''' and saving, since a flush occurs whenever the code windows are
        ''' displayed or there is a build. Neither of those items demands a save.
        Public Overloads Overrides Sub Flush()
            ' Nothing to flush if nothing has changed.
            If Not dirty Then
                Exit Sub
            End If

            ' We use an XmlDocument to build up the XML for
            ' the designer.  Here is a sample XML chunk:
            '
            Dim document As New XmlDocument()

            ' This element will serve as the undisputed DocumentElement (root)
            ' of our document. This allows us to have objects of equal level below,
            ' which we need, since component tray items are not children of the form.
            '
            document.AppendChild(document.CreateElement("DOCUMENT_ELEMENT"))

            ' We start with the root component and then continue
            ' to all the rest of the components.  The nametable
            ' object we create keeps track of which objects we have
            ' seen.  As we write out an object's contents, we save
            ' it in the nametable, so we don't write out an object
            ' twice.
            '
            Dim root As IComponent = host.RootComponent
            Dim nametable As New Hashtable(host.Container.Components.Count)

            document.DocumentElement.AppendChild(WriteObject(document, nametable, root))

            For Each comp As IComponent In host.Container.Components
                If comp IsNot root AndAlso Not nametable.ContainsKey(comp) Then
                    document.DocumentElement.AppendChild(WriteObject(document, nametable, comp))
                End If
            Next

            ' Along with the XML, we also represent the designer in a CodeCompileUnit,
            ' which we can use to generate C# and VB, and which we can compile from.
            '
            Dim code As New CodeCompileUnit()

            ' Our dummy namespace is the name of our main form + "Namespace". Creative, eh?
            Dim ns As New CodeNamespace(root.Site.Name & "Namespace")
            ns.[Imports].Add(New CodeNamespaceImport("System"))

            ' We need to look at our type resolution service to find out what references
            ' to import.
            '
            Dim strs As TypeResolutionService = TryCast(host.GetService(GetType(ITypeResolutionService)), TypeResolutionService)
            For Each assm As Assembly In strs.RefencedAssemblies
                ns.[Imports].Add(New CodeNamespaceImport(assm.GetName().Name))
            Next

            ' Autogenerates member declaration and InitializeComponent()
            ' in a new CodeTypeDeclaration
            '
            Dim a As RootDesignerSerializerAttribute = TryCast(TypeDescriptor.GetAttributes(root)(GetType(RootDesignerSerializerAttribute)), RootDesignerSerializerAttribute)
            Dim t As Type = host.[GetType](a.SerializerTypeName)
            Dim cds As CodeDomSerializer = TryCast(Activator.CreateInstance(t), CodeDomSerializer)
            Dim manager As IDesignerSerializationManager = TryCast(host.GetService(GetType(IDesignerSerializationManager)), IDesignerSerializationManager)
            Dim td As CodeTypeDeclaration = TryCast(cds.Serialize(manager, root), CodeTypeDeclaration)

            ' We need a constructor that will call the InitializeComponent()
            ' that we just generated.
            '
            Dim con As New CodeConstructor()
            con.Attributes = MemberAttributes.[Public]
            con.Statements.Add(New CodeMethodInvokeExpression(New CodeMethodReferenceExpression(New CodeThisReferenceExpression(), "InitializeComponent")))
            td.Members.Add(con)

            ' Finally our Main method, where the magic begins.
            Dim main As New CodeEntryPointMethod()
            main.Name = "Main"
            main.Attributes = MemberAttributes.[Public] Or MemberAttributes.[Static]
            main.CustomAttributes.Add(New CodeAttributeDeclaration("System.STAThreadAttribute"))
            main.Statements.Add(New CodeMethodInvokeExpression(New CodeMethodReferenceExpression(New CodeTypeReferenceExpression(GetType(System.Windows.Forms.Application)), "Run"), New CodeExpression() {New CodeObjectCreateExpression(New CodeTypeReference(root.Site.Name))}))
            td.Members.Add(main)


            ns.Types.Add(td)
            code.Namespaces.Add(ns)

            ' Now we reset our dirty bit and set the member
            ' variables.
            dirty = False
            xmlDocument = document
            codeCompileUnit = code

            ' Now we update the code windows to show what new stuff we've learned.
            UpdateCodeWindows()
        End Sub

        ''' This method writes out the contents of our designer in C#, VB, and XML.
        ''' For the first two, it generates code from our codeCompileUnit. For the XML,
        ''' it writes out the contents of xmlDocument.
        Private Sub UpdateCodeWindows()
            ' The main form's TabControl was added to the host's lists of services
            ' just so we could get at it here. Fortunately for us, each tab page
            ' has but one Control--a textbox. 
            '
            Dim tc As TabControl = TryCast(host.GetService(GetType(TabControl)), TabControl)
            Dim csWindow As TextBox = TryCast(tc.TabPages(1).Controls(0), TextBox)
            Dim vbWindow As TextBox = TryCast(tc.TabPages(2).Controls(0), TextBox)
            Dim xmlWindow As TextBox = TryCast(tc.TabPages(3).Controls(0), TextBox)

            ' The string writer we'll generate code to.
            Dim sw As StringWriter

            ' The options for our code generation.
            Dim o As New CodeGeneratorOptions()
            o.BlankLinesBetweenMembers = True
            o.BracingStyle = "C"
            o.ElseOnClosing = False
            o.IndentString = "    "

            ' CSharp Code Generation
            sw = New StringWriter()
            Dim cs As New CSharpCodeProvider()
            cs.CreateGenerator().GenerateCodeFromCompileUnit(codeCompileUnit, sw, o)
            csWindow.Text = sw.ToString()
            sw.Close()

            ' VB Code Generation
            sw = New StringWriter()
            Dim vb As New VBCodeProvider()
            vb.CreateGenerator().GenerateCodeFromCompileUnit(codeCompileUnit, sw, o)
            vbWindow.Text = sw.ToString()
            sw.Close()

            ' XML Output
            sw = New StringWriter()
            Dim xtw As New XmlTextWriter(sw)
            xtw.Formatting = Formatting.Indented
            xmlDocument.WriteTo(xtw)

            ' Get rid of our artificial super-root before we display the XML.
            '
            Dim cleanup As String = sw.ToString().Replace("<DOCUMENT_ELEMENT>", "")
            cleanup = cleanup.Replace("</DOCUMENT_ELEMENT>", "")
            xmlWindow.Text = cleanup
            sw.Close()
        End Sub

        ''' Simple helper method that returns true if the given type converter supports
        ''' two-way conversion of the given type.
        Private Function GetConversionSupported(ByVal converter As TypeConverter, ByVal conversionType As Type) As Boolean
            Return (converter.CanConvertFrom(conversionType) AndAlso converter.CanConvertTo(conversionType))
        End Function

        ' As soon as things change, we're dirty, so Flush()ing will give us a new
        ' xmlDocument and codeCompileUnit.
        Private Sub OnComponentChanged(ByVal sender As Object, ByVal ce As ComponentChangedEventArgs)
            dirty = True
            unsaved = True
        End Sub

        Private Sub OnComponentAddedRemoved(ByVal sender As Object, ByVal ce As ComponentEventArgs)
            dirty = True
            unsaved = True
        End Sub

        ''' This method prompts the user to see if it is OK to dispose this document.  
        ''' The prompt only happens if the user has made changes.
        Friend Function PromptDispose() As Boolean
            If dirty OrElse unsaved Then
                Select Case MessageBox.Show("Save changes to existing designer?", "Unsaved Changes", MessageBoxButtons.YesNoCancel)
                    Case DialogResult.Yes
                        Save(False)
                        Exit Select
                    Case DialogResult.Cancel
                        Return False
                End Select
            End If
            Return True
        End Function

        '''  Reads an Event node and binds the event.
        Private Sub ReadEvent(ByVal childNode As XmlNode, ByVal instance As Object, ByVal errors As ArrayList)
            Dim bindings As IEventBindingService = TryCast(host.GetService(GetType(IEventBindingService)), IEventBindingService)
            If bindings Is Nothing Then
                errors.Add("Unable to contact event binding service so we can't bind any events")
                Exit Sub
            End If

            Dim nameAttr As XmlAttribute = childNode.Attributes("name")
            If nameAttr Is Nothing Then
                errors.Add("No event name")
                Exit Sub
            End If

            Dim methodAttr As XmlAttribute = childNode.Attributes("method")
            If methodAttr Is Nothing OrElse methodAttr.Value Is Nothing OrElse methodAttr.Value.Length = 0 Then
                errors.Add(String.Format("Event {0} has no method bound to it"))
                Exit Sub
            End If

            Dim evt As EventDescriptor = TypeDescriptor.GetEvents(instance)(nameAttr.Value)
            If evt Is Nothing Then
                errors.Add(String.Format("Event {0} does not exist on {1}", nameAttr.Value, instance.[GetType]().FullName))
                Exit Sub
            End If

            Dim prop As PropertyDescriptor = bindings.GetEventProperty(evt)
            Debug.Assert(prop IsNot Nothing, "Bad event binding service")

            Try
                prop.SetValue(instance, methodAttr.Value)
            Catch ex As Exception
                errors.Add(ex.Message)
            End Try
        End Sub

        ''' This method is used to parse the given file.  Before calling this 
        ''' method the host member variable must be setup.  This method will
        ''' create a data set, read the data set from the XML stored in the
        ''' file, and then walk through the data set and create components
        ''' stored within it.  The data set is stored in the persistedData
        ''' member variable upon return.
        ''' 
        ''' This method never throws exceptions.  It will set the successful
        ''' ref parameter to false when there are catastrophic errors it can't
        ''' resolve (like being unable to parse the XML).  All error exceptions
        ''' are added to the errors array list, including minor errors.
        Private Function ReadFile(ByVal fileName As String, ByVal errors As ArrayList, ByRef document As XmlDocument) As String
            Dim baseClass As String = Nothing

            ' Anything unexpected is a fatal error.
            '
            Try
                ' The main form and items in the component tray will be at the
                ' same level, so we have to create a higher super-root in order
                ' to construct our XmlDocument.
                '
                Dim sr As New StreamReader(fileName)
                Dim cleandown As String = sr.ReadToEnd()
                cleandown = "<DOCUMENT_ELEMENT>" & cleandown & "</DOCUMENT_ELEMENT>"
                Dim doc As New XmlDocument()
                doc.LoadXml(cleandown)

                ' Now, walk through the document's elements.
                '
                For Each node As XmlNode In doc.DocumentElement.ChildNodes
                    If baseClass Is Nothing Then
                        baseClass = node.Attributes("name").Value
                    End If
                    If node.Name.Equals("Object") Then
                        ReadObject(node, errors)
                    Else
                        errors.Add(String.Format("Node type {0} is not allowed here.", node.Name))
                    End If
                Next

                document = doc
            Catch ex As Exception
                document = Nothing
                errors.Add(ex)
            End Try

            Return baseClass
        End Function

        Private Function ReadInstanceDescriptor(ByVal node As XmlNode, ByVal errors As ArrayList) As Object
            ' First, need to deserialize the member
            '
            Dim memberAttr As XmlAttribute = node.Attributes("member")
            If memberAttr Is Nothing Then
                errors.Add("No member attribute on instance descriptor")
                Return Nothing
            End If

            Dim data As Byte() = Convert.FromBase64String(memberAttr.Value)
            Dim formatter As New BinaryFormatter()
            Dim stream As New MemoryStream(data)
            Dim mi As MemberInfo = DirectCast(formatter.Deserialize(stream), MemberInfo)
            Dim args As Object() = Nothing

            ' Check to see if this member needs arguments.  If so, gather
            ' them from the XML.
            If TypeOf mi Is MethodBase Then
                Dim paramInfos As ParameterInfo() = DirectCast(mi, MethodBase).GetParameters()
                args = New Object(paramInfos.Length - 1) {}
                Dim idx As Integer = 0

                For Each child As XmlNode In node.ChildNodes
                    If child.Name.Equals("Argument") Then
                        Dim value As Object
                        If Not ReadValue(child, TypeDescriptor.GetConverter(paramInfos(idx).ParameterType), errors, value) Then
                            Return Nothing
                        End If

                        args(System.Math.Max(System.Threading.Interlocked.Increment(idx), idx - 1)) = value
                    End If
                Next

                If idx <> paramInfos.Length Then
                    errors.Add(String.Format("Member {0} requires {1} arguments, not {2}.", mi.Name, args.Length, idx))
                    Return Nothing
                End If
            End If

            Dim id As New InstanceDescriptor(mi, args)
            Dim instance As Object = id.Invoke()

            ' Ok, we have our object.  Now, check to see if there are any properties, and if there are, 
            ' set them.
            '
            For Each prop As XmlNode In node.ChildNodes
                If prop.Name.Equals("Property") Then
                    ReadProperty(prop, instance, errors)
                End If
            Next

            Return instance
        End Function

        ''' Reads the "Object" tags. This returns an instance of the
        ''' newly created object. Returns null if there was an error.
        Private Function ReadObject(ByVal node As XmlNode, ByVal errors As ArrayList) As Object
            Dim typeAttr As XmlAttribute = node.Attributes("type")
            If typeAttr Is Nothing Then
                errors.Add("<Object> tag is missing required type attribute")
                Return Nothing
            End If

            Dim type__1 As Type = Type.[GetType](typeAttr.Value)
            If type__1 Is Nothing Then
                errors.Add(String.Format("Type {0} could not be loaded.", typeAttr.Value))
                Return Nothing
            End If

            ' This can be null if there is no name for the object.
            '
            Dim nameAttr As XmlAttribute = node.Attributes("name")

            Dim instance As Object

            If GetType(IComponent).IsAssignableFrom(type__1) Then
                If nameAttr Is Nothing Then
                    instance = host.CreateComponent(type__1)
                Else
                    instance = host.CreateComponent(type__1, nameAttr.Value)
                End If
            Else
                instance = Activator.CreateInstance(type__1)
            End If

            ' Got an object, now we must process it.  Check to see if this tag
            ' offers a child collection for us to add children to.
            '
            Dim childAttr As XmlAttribute = node.Attributes("children")
            Dim childList As IList = Nothing
            If childAttr IsNot Nothing Then
                Dim childProp As PropertyDescriptor = TypeDescriptor.GetProperties(instance)(childAttr.Value)
                If childProp Is Nothing Then
                    errors.Add(String.Format("The children attribute lists {0} as the child collection but this is not a property on {1}", childAttr.Value, instance.[GetType]().FullName))
                Else
                    childList = TryCast(childProp.GetValue(instance), IList)
                    If childList Is Nothing Then
                        errors.Add(String.Format("The property {0} was found but did not return a valid IList", childProp.Name))
                    End If
                End If
            End If

            ' Now, walk the rest of the tags under this element.
            '
            For Each childNode As XmlNode In node.ChildNodes
                If childNode.Name.Equals("Object") Then
                    ' Another object.  In this case, create the object, and
                    ' parent it to ours using the children property.  If there
                    ' is no children property, bail out now.
                    If childAttr Is Nothing Then
                        errors.Add("Child object found but there is no children attribute")
                        Continue For
                    End If

                    ' no sense doing this if there was an error getting the property.  We've already reported the
                    ' error above.
                    If childList IsNot Nothing Then
                        Dim childInstance As Object = ReadObject(childNode, errors)
                        childList.Add(childInstance)
                    End If
                ElseIf childNode.Name.Equals("Property") Then
                    ' A property.  Ask the property to parse itself.
                    '
                    ReadProperty(childNode, instance, errors)
                ElseIf childNode.Name.Equals("Event") Then
                    ' An event.  Ask the event to parse itself.
                    '
                    ReadEvent(childNode, instance, errors)
                End If
            Next
            Return instance
        End Function

        ''' Parses the given XML node and sets the resulting property value.
        Private Sub ReadProperty(ByVal node As XmlNode, ByVal instance As Object, ByVal errors As ArrayList)
            Dim nameAttr As XmlAttribute = node.Attributes("name")
            If nameAttr Is Nothing Then
                errors.Add("Property has no name")
                Exit Sub
            End If

            Dim prop As PropertyDescriptor = TypeDescriptor.GetProperties(instance)(nameAttr.Value)
            If prop Is Nothing Then
                errors.Add(String.Format("Property {0} does not exist on {1}", nameAttr.Value, instance.[GetType]().FullName))
                Exit Sub
            End If

            ' Get the type of this property.  We have three options:
            ' 1.  A normal read/write property.
            ' 2.  A "Content" property.
            ' 3.  A collection.
            '
            Dim isContent As Boolean = prop.Attributes.Contains(DesignerSerializationVisibilityAttribute.Content)

            If isContent Then
                Dim value As Object = prop.GetValue(instance)

                ' Handle the case of a content property that is a collection.
                '
                If TypeOf value Is IList Then
                    For Each child As XmlNode In node.ChildNodes
                        If child.Name.Equals("Item") Then
                            Dim item As Object
                            Dim typeAttr As XmlAttribute = child.Attributes("type")
                            If typeAttr Is Nothing Then
                                errors.Add("Item has no type attribute")
                                Continue For
                            End If

                            Dim type__1 As Type = Type.[GetType](typeAttr.Value)
                            If type__1 Is Nothing Then
                                errors.Add(String.Format("Item type {0} could not be found.", typeAttr.Value))
                                Continue For
                            End If

                            If ReadValue(child, TypeDescriptor.GetConverter(type__1), errors, item) Then
                                Try
                                    DirectCast(value, IList).Add(item)
                                Catch ex As Exception
                                    errors.Add(ex.Message)
                                End Try
                            End If
                        Else
                            errors.Add(String.Format("Only Item elements are allowed in collections, not {0} elements.", child.Name))
                        End If
                    Next
                Else
                    ' Handle the case of a content property that consists of child properties.
                    '
                    For Each child As XmlNode In node.ChildNodes
                        If child.Name.Equals("Property") Then
                            ReadProperty(child, value, errors)
                        Else
                            errors.Add(String.Format("Only Property elements are allowed in content properties, not {0} elements.", child.Name))
                        End If
                    Next
                End If
            Else
                Dim value As Object
                If ReadValue(node, prop.Converter, errors, value) Then
                    ' ReadValue succeeded.  Fill in the property value.
                    '
                    Try
                        prop.SetValue(instance, value)
                    Catch ex As Exception
                        errors.Add(ex.Message)
                    End Try
                End If
            End If
        End Sub

        ''' Generic function to read an object value.  Returns true if the read
        ''' succeeded.
        Private Function ReadValue(ByVal node As XmlNode, ByVal converter As TypeConverter, ByVal errors As ArrayList, ByRef value As Object) As Boolean
            Try
                For Each child As XmlNode In node.ChildNodes
                    If child.NodeType = XmlNodeType.Text Then
                        value = converter.ConvertFromInvariantString(node.InnerText)
                        Return True
                    ElseIf child.Name.Equals("Binary") Then
                        Dim data As Byte() = Convert.FromBase64String(child.InnerText)

                        ' Binary blob.  Now, check to see if the type converter
                        ' can convert it.  If not, use serialization.
                        '
                        If GetConversionSupported(converter, GetType(Byte())) Then
                            value = converter.ConvertFrom(Nothing, CultureInfo.InvariantCulture, data)
                            Return True
                        Else
                            Dim formatter As New BinaryFormatter()
                            Dim stream As New MemoryStream(data)
                            value = formatter.Deserialize(stream)
                            Return True
                        End If
                    ElseIf child.Name.Equals("InstanceDescriptor") Then
                        value = ReadInstanceDescriptor(child, errors)
                        Return (value IsNot Nothing)
                    Else
                        errors.Add(String.Format("Unexpected element type {0}", child.Name))
                        value = Nothing
                        Return False
                    End If
                Next

                ' If we get here, it is because there were no nodes.  No nodes and no inner
                ' text is how we signify null.
                '
                value = Nothing
                Return True
            Catch ex As Exception
                errors.Add(ex.Message)
                value = Nothing
                Return False
            End Try
        End Function

        ''' This method writes a given byte[] into the XML document, returning the node that
        ''' it just created.  Byte arrays have the following XML:
        ''' 
        ''' <c>
        ''' <Binary>
        '''		64 bit encoded string representing binary data
        ''' </Binary>
        ''' </c>
        Private Function WriteBinary(ByVal document As XmlDocument, ByVal value As Byte()) As XmlNode
            Dim node As XmlNode = document.CreateElement("Binary")
            node.InnerText = Convert.ToBase64String(value)
            Return node
        End Function

        ''' Writes the given IList contents into the given parent node.
        Private Sub WriteCollection(ByVal document As XmlDocument, ByVal list As IList, ByVal parent As XmlNode)
            For Each obj As Object In list
                Dim node As XmlNode = document.CreateElement("Item")
                Dim typeAttr As XmlAttribute = document.CreateAttribute("type")
                typeAttr.Value = obj.[GetType]().AssemblyQualifiedName
                node.Attributes.Append(typeAttr)
                WriteValue(document, obj, node)
                parent.AppendChild(node)
            Next
        End Sub

        ''' This method writes a given instance descriptor into the XML document, returning a node
        ''' that it just created.  Instance descriptors have the following XML:
        ''' 
        ''' <c>
        ''' <InstanceDescriptor member="asdfasdfasdf">
        '''		<Object>
        '''			// param value
        '''		</Object>
        ''' </InstanceDescriptor>
        ''' </c>
        ''' 
        ''' Here, member is a 64 bit encoded string representing the member, and there is one Parameter
        ''' tag for each parameter of the descriptor.
        Private Function WriteInstanceDescriptor(ByVal document As XmlDocument, ByVal desc As InstanceDescriptor, ByVal value As Object) As XmlNode
            Dim node As XmlNode = document.CreateElement("InstanceDescriptor")
            Dim formatter As New BinaryFormatter()
            Dim stream As New MemoryStream()
            formatter.Serialize(stream, desc.MemberInfo)
            Dim memberAttr As XmlAttribute = document.CreateAttribute("member")
            memberAttr.Value = Convert.ToBase64String(stream.ToArray())
            node.Attributes.Append(memberAttr)

            For Each arg As Object In desc.Arguments
                Dim argNode As XmlNode = document.CreateElement("Argument")
                If WriteValue(document, arg, argNode) Then
                    node.AppendChild(argNode)
                End If
            Next

            ' Instance descriptors also support "partial" creation, where 
            ' properties must also be persisted.
            '
            If Not desc.IsComplete Then
                Dim props As PropertyDescriptorCollection = TypeDescriptor.GetProperties(value, propertyAttributes)
                WriteProperties(document, props, value, node, "Property")
            End If

            Return node
        End Function

        ''' This method writes the given object out to the XML document.  Objects have
        ''' the following XML:
        ''' 
        ''' <c>
        ''' Object type="<object type>" name="<object name>" children="<child property name>"
        ''' 
        ''' </Object>
        ''' </c>
        ''' 
        ''' Here, Object is the element that defines a custom object.  Type is required
        ''' and specifies the data type of the object.  Name is optional.  If present, it names
        ''' this object, adding it to the container if the object is an IComponent.
        ''' Finally, the children attribute is optional.  If present, this object can have
        ''' nested objects, and those objects will be added to the given property name.  The
        ''' property must be a collection property that returns an object that implements IList.
        ''' 
        ''' Inside the object tag there can be zero or more of the following subtags:
        ''' 
        '''		InstanceDescriptor -- describes how to create an instance of the object.
        '''		Property -- a property set on the object
        '''		Event -- an event binding
        '''		Binary -- binary data
        Private Function WriteObject(ByVal document As XmlDocument, ByVal nametable As IDictionary, ByVal value As Object) As XmlNode
            Debug.Assert(value IsNot Nothing, "Should not invoke WriteObject with a null value")

            Dim node As XmlNode = document.CreateElement("Object")

            Dim typeAttr As XmlAttribute = document.CreateAttribute("type")
            typeAttr.Value = value.[GetType]().AssemblyQualifiedName
            node.Attributes.Append(typeAttr)

            ' Does this object have a name?
            '
            Dim component As IComponent = TryCast(value, IComponent)
            If component IsNot Nothing AndAlso component.Site IsNot Nothing AndAlso component.Site.Name IsNot Nothing Then
                Dim nameAttr As XmlAttribute = document.CreateAttribute("name")
                nameAttr.Value = component.Site.Name
                node.Attributes.Append(nameAttr)
                Debug.Assert(nametable(component) Is Nothing, "WriteObject should not be called more than once for the same object.  Use WriteReference instead")
                nametable(value) = component.Site.Name
            End If

            ' Special case:  We want Windows Forms controls to "nest", so child
            ' elements are child controls on the form.  This requires either an
            ' extensible serialization mechanism (like the existing CodeDom
            ' serialization scheme), or it requires special casing in the
            ' serialization code.  We choose the latter in order to make
            ' this example easier to understand.
            '
            Dim isControl As Boolean = (TypeOf value Is Control)

            If isControl Then
                Dim childAttr As XmlAttribute = document.CreateAttribute("children")
                childAttr.Value = "Controls"
                node.Attributes.Append(childAttr)
            End If

            If component IsNot Nothing Then
                ' We have a component.  Write out the definition for the component here.  If the
                ' component is also a control, recurse so we build up the parent hierarchy.
                '
                If isControl Then
                    For Each child As Control In DirectCast(value, Control).Controls
                        If child.Site IsNot Nothing AndAlso child.Site.Container Is host.Container Then
                            node.AppendChild(WriteObject(document, nametable, child))
                        End If
                    Next
                End If

                ' Now do our own properties.
                '
                Dim properties As PropertyDescriptorCollection = TypeDescriptor.GetProperties(value, propertyAttributes)

                If isControl Then
                    ' If we are a control and we can locate the control property, we should remove
                    ' the property from the collection. The collection that comes back from TypeDescriptor
                    ' is read-only, however, so we must clone it first.
                    '
                    Dim controlProp As PropertyDescriptor = properties("Controls")
                    If controlProp IsNot Nothing Then
                        Dim propArray As PropertyDescriptor() = New PropertyDescriptor(properties.Count - 2) {}
                        Dim idx As Integer = 0
                        For Each p As PropertyDescriptor In properties
                            If p IsNot controlProp Then
                                propArray(System.Math.Max(System.Threading.Interlocked.Increment(idx), idx - 1)) = p
                            End If
                        Next

                        properties = New PropertyDescriptorCollection(propArray)
                    End If
                End If

                WriteProperties(document, properties, value, node, "Property")

                Dim events As EventDescriptorCollection = TypeDescriptor.GetEvents(value, propertyAttributes)
                Dim bindings As IEventBindingService = TryCast(host.GetService(GetType(IEventBindingService)), IEventBindingService)
                If bindings IsNot Nothing Then
                    properties = bindings.GetEventProperties(events)
                    WriteProperties(document, properties, value, node, "Event")
                End If
            Else
                ' Not a component, so we just write out the value.
                '
                WriteValue(document, value, node)
            End If

            Return node
        End Function

        ''' This method writes zero or more property elements into the given parent node.  
        Private Sub WriteProperties(ByVal document As XmlDocument, ByVal properties As PropertyDescriptorCollection, ByVal value As Object, ByVal parent As XmlNode, ByVal elementName As String)
            For Each prop As PropertyDescriptor In properties
                If prop.Name = "AutoScaleBaseSize" Then
                    Dim _DEBUG_ As String = prop.Name
                End If

                If prop.ShouldSerializeValue(value) Then
                    Dim node As XmlNode = document.CreateElement(elementName)
                    Dim attr As XmlAttribute = document.CreateAttribute("name")
                    attr.Value = prop.Name
                    node.Attributes.Append(attr)

                    Dim visibility As DesignerSerializationVisibilityAttribute = DirectCast(prop.Attributes(GetType(DesignerSerializationVisibilityAttribute)), DesignerSerializationVisibilityAttribute)
                    Select Case visibility.Visibility
                        Case DesignerSerializationVisibility.Visible
                            If Not prop.IsReadOnly AndAlso WriteValue(document, prop.GetValue(value), node) Then
                                parent.AppendChild(node)
                            End If
                            Exit Select

                        Case DesignerSerializationVisibility.Content
                            ' A "Content" property needs to have its properties stored here, not the actual value.  We 
                            ' do another special case here to account for collections.  Collections are content properties
                            ' that implement IList and are read-only.
                            '
                            Dim propValue As Object = prop.GetValue(value)

                            If GetType(IList).IsAssignableFrom(prop.PropertyType) Then
                                WriteCollection(document, DirectCast(propValue, IList), node)
                            Else
                                Dim props As PropertyDescriptorCollection = TypeDescriptor.GetProperties(propValue, propertyAttributes)
                                WriteProperties(document, props, propValue, node, elementName)
                            End If
                            If node.ChildNodes.Count > 0 Then
                                parent.AppendChild(node)
                            End If
                            Exit Select
                        Case Else

                            Exit Select
                    End Select
                End If
            Next
        End Sub

        ''' Writes a reference to the given component.  Emits the following
        ''' XML:
        ''' 
        ''' <c>
        ''' <Reference name="component name"></Reference>
        ''' </c>
        Private Function WriteReference(ByVal document As XmlDocument, ByVal value As IComponent) As XmlNode
            Debug.Assert(value IsNot Nothing AndAlso value.Site IsNot Nothing AndAlso value.Site.Container Is host.Container, "Invalid component passed to WriteReference")

            Dim node As XmlNode = document.CreateElement("Reference")
            Dim attr As XmlAttribute = document.CreateAttribute("name")
            attr.Value = value.Site.Name
            node.Attributes.Append(attr)
            Return node
        End Function

        ''' This method writes the given object into the given parent node.  It returns
        ''' true if it was successful, or false if it was unable to convert the object
        ''' to XML.
        Private Function WriteValue(ByVal document As XmlDocument, ByVal value As Object, ByVal parent As XmlNode) As Boolean
            ' For empty values, we just return.  This creates an empty node.
            If value Is Nothing Then
                Return True
            End If

            Dim converter As TypeConverter = TypeDescriptor.GetConverter(value)

            If GetConversionSupported(converter, GetType(String)) Then
                ' Strings have the most fidelity.  If this object
                ' supports being converted to a string, do so, and then
                ' we're done.
                '
                parent.InnerText = DirectCast(converter.ConvertTo(Nothing, CultureInfo.InvariantCulture, value, GetType(String)), String)
            ElseIf GetConversionSupported(converter, GetType(Byte())) Then
                ' Binary blobs are converted by encoding as a binary element.
                ' 
                Dim data As Byte() = DirectCast(converter.ConvertTo(Nothing, CultureInfo.InvariantCulture, value, GetType(Byte())), Byte())
                parent.AppendChild(WriteBinary(document, data))
            ElseIf GetConversionSupported(converter, GetType(InstanceDescriptor)) Then
                ' InstanceDescriptors are encoded as an InstanceDescriptor element.
                '
                Dim id As InstanceDescriptor = DirectCast(converter.ConvertTo(Nothing, CultureInfo.InvariantCulture, value, GetType(InstanceDescriptor)), InstanceDescriptor)
                parent.AppendChild(WriteInstanceDescriptor(document, id, value))
            ElseIf TypeOf value Is IComponent AndAlso DirectCast(value, IComponent).Site IsNot Nothing AndAlso DirectCast(value, IComponent).Site.Container Is host.Container Then
                ' IComponent.  Treat this as a reference.
                '
                parent.AppendChild(WriteReference(document, DirectCast(value, IComponent)))
            ElseIf value.[GetType]().IsSerializable Then
                ' Finally, check to see if this object is serializable.  If it is, we serialize it here
                ' and then write it as a binary.
                '
                Dim formatter As New BinaryFormatter()
                Dim stream As New MemoryStream()
                formatter.Serialize(stream, value)
                Dim binaryNode As XmlNode = WriteBinary(document, stream.ToArray())
                parent.AppendChild(binaryNode)
            Else
                Return False
            End If
            Return True
        End Function

        Public Function GetService(ByVal serviceType As Type) As Object
            Return host.GetService(serviceType)
        End Function

        Public ReadOnly Property LoaderHost() As IDesignerLoaderHost
            Get
                Return TryCast(host, IDesignerLoaderHost)
            End Get
        End Property

        ''' Save the current state of the loader. If the user loaded the file
        ''' or saved once before, then he doesn't need to select a file again.
        ''' Unless this is being called as a result of "Save As..." being clicked,
        ''' in which case forceFilePrompt will be true.
        Friend Sub Save(ByVal forceFilePrompt As Boolean)
            Try
                If dirty Then
                    ' Flush any changes to the buffer.
                    Flush()
                End If


                ' If the buffer has no name or this is a "Save As...",
                ' prompt the user for a file name. The user can save
                ' either the C#, VB, or XML (though only the XML can be loaded).
                '
                Dim filterIndex As Integer = 3
                If (fileName Is Nothing) OrElse forceFilePrompt Then
                    Dim dlg As New SaveFileDialog()
                    dlg.DefaultExt = "xml"
                    dlg.Filter = "C# Files|*.cs|Visual Basic Files|*.vb|XML Files|*.xml"

                    If dlg.ShowDialog() = DialogResult.OK Then
                        fileName = dlg.FileName
                        filterIndex = dlg.FilterIndex
                    End If
                End If

                If fileName IsNot Nothing Then
                    Select Case filterIndex
                        Case 1
                            If True Then
                                ' Generate C# code from our codeCompileUnit and save it.
                                Dim o As New CodeGeneratorOptions()
                                o.BlankLinesBetweenMembers = True
                                o.BracingStyle = "C"
                                o.ElseOnClosing = False
                                o.IndentString = "    "
                                Dim sw As New StreamWriter(fileName)
                                Dim cs As New CSharpCodeProvider()
                                cs.CreateGenerator().GenerateCodeFromCompileUnit(codeCompileUnit, sw, o)
                                sw.Close()
                            End If
                            Exit Select
                        Case 2
                            If True Then
                                ' Generate VB code from our codeCompileUnit and save it.
                                Dim o As New CodeGeneratorOptions()
                                o.BlankLinesBetweenMembers = True
                                o.BracingStyle = "C"
                                o.ElseOnClosing = False
                                o.IndentString = "    "
                                Dim sw As New StreamWriter(fileName)
                                Dim vb As New VBCodeProvider()
                                vb.CreateGenerator().GenerateCodeFromCompileUnit(codeCompileUnit, sw, o)
                                sw.Close()
                            End If
                            Exit Select
                        Case 3
                            If True Then
                                ' Write out our xmlDocument to a file.
                                Dim sw As New StringWriter()
                                Dim xtw As New XmlTextWriter(sw)
                                xtw.Formatting = Formatting.Indented
                                xmlDocument.WriteTo(xtw)

                                ' Get rid of our artificial super-root before we save out
                                ' the XML.
                                '
                                Dim cleanup As String = sw.ToString().Replace("<DOCUMENT_ELEMENT>", "")
                                cleanup = cleanup.Replace("</DOCUMENT_ELEMENT>", "")
                                xtw.Close()
                                Dim file As New StreamWriter(fileName)
                                file.Write(cleanup)
                                file.Close()
                            End If
                            Exit Select
                    End Select
                    unsaved = False
                End If
            Catch ex As Exception
                MessageBox.Show("Error during save: " & ex.Message)
            End Try
        End Sub

        ''' Called when we want to build an executable. Returns true if we succeeded.
        Friend Function Build() As Boolean
            If dirty Then
                ' Flush any changes made to the buffer.
                Flush()
            End If

            ' If we haven't already chosen a spot to write the executable to,
            ' do so now.
            '
            If executable Is Nothing Then
                Dim dlg As New SaveFileDialog()
                dlg.DefaultExt = "exe"
                dlg.Filter = "Executables|*.exe"

                If dlg.ShowDialog() = DialogResult.OK Then
                    executable = dlg.FileName
                End If
            End If

            If executable IsNot Nothing Then
                ' We'll need our type resolution service in order to find out what
                ' assemblies we're dealing with.
                '
                Dim strs As TypeResolutionService = TryCast(host.GetService(GetType(ITypeResolutionService)), TypeResolutionService)

                ' We need to collect the parameters that our compiler will use.
                Dim cp As New CompilerParameters()

                ' First, we tell our compiler to reference the assemblies which
                ' our designers have referenced (the ones which have import statements
                ' in our codeCompileUnit).....
                '
                For Each assm As Assembly In strs.RefencedAssemblies
                    cp.ReferencedAssemblies.Add(assm.Location)

                    ' .....then we have to look at each one of those assemblies,
                    ' see which assemblies they reference, and make sure our compiler
                    ' references those too! Phew!
                    '
                    For Each refAssmName As AssemblyName In assm.GetReferencedAssemblies()
                        Dim refAssm As Assembly = Assembly.Load(refAssmName)
                        cp.ReferencedAssemblies.Add(refAssm.Location)
                    Next
                Next

                cp.GenerateExecutable = True
                cp.OutputAssembly = executable

                ' Remember our main class is not Form, but Form1 (or whatever the user calls it)!
                cp.MainClass = (host.RootComponent.Site.Name & "Namespace.") + host.RootComponent.Site.Name
                Dim cc As ICodeCompiler = New CSharpCodeProvider().CreateCompiler()
                Dim cr As CompilerResults = cc.CompileAssemblyFromDom(cp, codeCompileUnit)
                If cr.Errors.HasErrors Then
                    Dim errors As String = ""
                    For Each [error] As CompilerError In cr.Errors
                        errors += [error].ErrorText & vbLf
                    Next
                    MessageBox.Show(errors, "Errors during compile.")
                End If
                Return Not cr.Errors.HasErrors
            End If

            Return False
        End Function

        ' Here we build the executable and then run it. We make sure to not start
        ' two of the same process.
        Friend Sub RunBuild()
            If (run Is Nothing) OrElse (run.HasExited) Then
                If Build() Then
                    run = New Process()
                    run.StartInfo.FileName = executable
                    run.Start()
                End If
            End If
        End Sub

        ' Just in case the red X in the upper right isn't good enough,
        ' we can kill our process here.
        Friend Sub [Stop]()
            If (run IsNot Nothing) AndAlso (Not run.HasExited) Then
                run.Kill()
            End If
        End Sub
    End Class
End Namespace
